"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.markCommonFields = exports.isCommonField = exports.getStaticPosition = exports.getPosition = exports.getOnOption = exports.getLookupFields = exports.getFullCommandMnemonics = exports.createEnrichedGetByType = exports.createEnrichedContext = void 0;
var _i18n = require("@kbn/i18n");
var _location = require("../../../ast/location");
var _is = require("../../../ast/is");
var _utils = require("../../../definitions/utils");
var mutate = _interopRequireWildcard(require("../../../mutate"));
var _leaf_printer = require("../../../pretty_print/leaf_printer");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const REGEX = /^(?<type>\w+((?<after_type>\s+((?<mnemonic>(JOIN|JOI|JO|J)((?<after_mnemonic>\s+((?<index>\S+((?<after_index>\s+(?<as>(AS|A))?(?<after_as>\s+(((?<alias>\S+)?(?<after_alias>\s+)?)?))?((?<on>(ON|O))?))?))?))?))?))?))?/i;
const positions = ['on', 'after_alias', 'alias', 'after_as', 'as', 'after_index', 'index', 'after_mnemonic', 'mnemonic', 'after_type', 'type'];
const getFullCommandMnemonics = command => {
  var _command$metadata$typ;
  const types = (_command$metadata$typ = command.metadata.types) !== null && _command$metadata$typ !== void 0 ? _command$metadata$typ : [];
  if (!types.length) {
    return [[command.name, command.metadata.description]];
  }
  return types.map(type => {
    var _type$description;
    return [`${type.name.toUpperCase()} ${command.name.toUpperCase()}`, (_type$description = type.description) !== null && _type$description !== void 0 ? _type$description : command.metadata.description];
  });
};

// facilitates fast checks for the existence of fields in the lookup index
// by caching the fields of the last lookup index pattern
exports.getFullCommandMnemonics = getFullCommandMnemonics;
const lookupIndexFieldSet = {
  set: new Set(),
  key: ''
};
const getLookupFields = async (command, getColumnsForQuery, context) => {
  if (!context) {
    return [];
  }
  const summary = mutate.commands.join.summarizeCommand(command);
  const joinIndexPattern = _leaf_printer.LeafPrinter.print(summary.target.index);
  const columns = await getColumnsForQuery(`FROM ${joinIndexPattern}`);
  if (lookupIndexFieldSet.key !== joinIndexPattern) {
    lookupIndexFieldSet.set = new Set(columns.map(c => c.name));
    lookupIndexFieldSet.key = joinIndexPattern;
  }
  return columns;
};

/** Returns the position based on regex matching. */
exports.getLookupFields = getLookupFields;
const getStaticPosition = text => {
  const match = text.match(REGEX);
  if (!match || !match.groups) {
    return 'none';
  }
  let pos = 'none';
  for (const position of positions) {
    if (match.groups[position]) {
      pos = position;
      break;
    }
  }
  return pos;
};
exports.getStaticPosition = getStaticPosition;
const getOnOption = command => {
  var _command$args;
  return (_command$args = command.args) === null || _command$args === void 0 ? void 0 : _command$args.find(arg => (0, _is.isOptionNode)(arg) && arg.name === 'on');
};
exports.getOnOption = getOnOption;
const getPosition = (text, command, cursorPosition) => {
  const pos = getStaticPosition(text);
  const joinCommand = command;
  const onOption = getOnOption(joinCommand);
  if (onOption) {
    const expressions = onOption.args;

    // No expressions yet or starting new expression after comma
    if (expressions.length === 0 || /,\s*$/.test(text)) {
      return {
        pos: 'on_expression',
        expression: undefined,
        isExpressionComplete: false
      };
    }
    const lastExpression = expressions[expressions.length - 1];

    // Cursor within incomplete expression
    if (lastExpression !== null && lastExpression !== void 0 && lastExpression.incomplete && lastExpression.location && (0, _location.within)(cursorPosition, lastExpression)) {
      return {
        pos: 'on_expression',
        expression: lastExpression,
        isExpressionComplete: false
      };
    }

    // Cursor within any complete expression
    for (const expr of expressions) {
      if (expr.location && (0, _location.within)(cursorPosition, expr)) {
        return {
          pos: 'on_expression',
          expression: expr,
          isExpressionComplete: !expr.incomplete
        };
      }
    }

    // Cursor after all expressions
    return {
      pos: 'on_expression',
      expression: lastExpression,
      isExpressionComplete: !(lastExpression !== null && lastExpression !== void 0 && lastExpression.incomplete)
    };
  }
  return {
    pos
  };
};

/**
 * Identifies common fields between source and lookup suggestions and marks them appropriately.
 * Common fields are those that exist in both source and lookup with the same label.

 */
exports.getPosition = getPosition;
const markCommonFields = (sourceSuggestions, lookupSuggestions) => {
  const sourceLabels = new Set(sourceSuggestions.map(({
    label
  }) => label));
  const commonFieldLabels = new Set(lookupSuggestions.map(({
    label
  }) => label).filter(label => sourceLabels.has(label)));

  // Mark common fields in source suggestions
  const markedSourceSuggestions = sourceSuggestions.map(suggestion => {
    if (commonFieldLabels.has(suggestion.label)) {
      let detail = suggestion.detail || '';
      if (detail) {
        detail += ' ';
      }
      detail += _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.join.commonFieldNote', {
        defaultMessage: '(common field)'
      });
      return {
        ...suggestion,
        sortText: '1-' + (suggestion.sortText || suggestion.label),
        detail,
        documentation: {
          value: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.join.sharedField', {
            defaultMessage: 'Field shared between the source and the lookup index'
          })
        }
      };
    }
    return suggestion;
  });

  // Filter out duplicate lookup fields
  const uniqueLookupSuggestions = lookupSuggestions.filter(({
    label
  }) => !commonFieldLabels.has(label));
  return {
    markedSourceSuggestions,
    uniqueLookupSuggestions,
    commonFieldLabels
  };
};

/** Creates an enriched context that includes lookup table fields in the columns map. */
exports.markCommonFields = markCommonFields;
const createEnrichedContext = async (originalContext, joinCommand, getColumnsForQuery) => {
  if (!originalContext) {
    return undefined;
  }
  const lookupFields = await getLookupFields(joinCommand, getColumnsForQuery, originalContext);
  const enrichedColumns = new Map(originalContext.columns);
  for (const field of lookupFields) {
    if (!enrichedColumns.has(field.name)) {
      enrichedColumns.set(field.name, field);
    }
  }
  return {
    ...originalContext,
    columns: enrichedColumns
  };
};

/**
 * Creates an enriched getByType function that includes lookup table fields
 * in addition to the source table fields.
 */
exports.createEnrichedContext = createEnrichedContext;
const createEnrichedGetByType = async (originalGetByType, joinCommand, getColumnsForQuery, context) => {
  const lookupFields = await getLookupFields(joinCommand, getColumnsForQuery, context);

  // Return wrapper function
  return async (type, ignored, options) => {
    const sourceColumns = await originalGetByType(type, ignored, options);
    const types = Array.isArray(type) ? type : [type];
    const filteredLookupFields = lookupFields.filter(({
      name,
      type: t
    }) => {
      return !(ignored !== null && ignored !== void 0 && ignored.includes(name)) && (types[0] === 'any' || types.includes(t));
    });
    const lookupSuggestions = (0, _utils.buildFieldsDefinitionsWithMetadata)(filteredLookupFields, [], options, context === null || context === void 0 ? void 0 : context.variables);

    // Use the utility function to mark common fields
    const {
      markedSourceSuggestions,
      uniqueLookupSuggestions
    } = markCommonFields(sourceColumns, lookupSuggestions);
    return [...markedSourceSuggestions, ...uniqueLookupSuggestions];
  };
};

// Check if a field is common (exists in both source and lookup tables)
exports.createEnrichedGetByType = createEnrichedGetByType;
const isCommonField = (fieldName, context) => {
  if (!(context !== null && context !== void 0 && context.columns)) {
    return false;
  }
  return context.columns.has(fieldName) && lookupIndexFieldSet.set.has(fieldName);
};
exports.isCommonField = isCommonField;