"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autocomplete = autocomplete;
var _i18n = require("@kbn/i18n");
var _map_expression = require("../../../definitions/utils/autocomplete/map_expression");
var _ = require("../../../..");
var _is = require("../../../ast/is");
var _types = require("../../../definitions/types");
var _helpers = require("../../../definitions/utils/autocomplete/helpers");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
var FusePosition = /*#__PURE__*/function (FusePosition) {
  FusePosition["BEFORE_NEW_ARGUMENT"] = "before_new_argument";
  FusePosition["SCORE_BY"] = "score_by";
  FusePosition["KEY_BY"] = "key_by";
  FusePosition["GROUP_BY"] = "group_by";
  FusePosition["WITH"] = "with";
  return FusePosition;
}(FusePosition || {});
function getPosition(innerText, command) {
  const {
    scoreBy,
    keyBy,
    groupBy,
    withOption
  } = (0, _utils.extractFuseArgs)(command);
  if (scoreBy && scoreBy.incomplete || (0, _utils.immediatelyAfterOptionField)(innerText, 'score by')) {
    return FusePosition.SCORE_BY;
  }
  if (groupBy && groupBy.incomplete || (0, _utils.immediatelyAfterOptionField)(innerText, 'group by')) {
    return FusePosition.GROUP_BY;
  }
  if (keyBy && keyBy.incomplete || (0, _utils.immediatelyAfterOptionFieldsList)(innerText, 'key by') || keyBy !== null && keyBy !== void 0 && keyBy.text.includes(_.EDITOR_MARKER)) {
    return FusePosition.KEY_BY;
  }
  if (withOption && withOption.incomplete) {
    return FusePosition.WITH;
  }
  return FusePosition.BEFORE_NEW_ARGUMENT;
}

// FUSE <fuse_method> SCORE BY <score_column> GROUP BY <group_column> KEY BY <key_columns> WITH <options>
async function autocomplete(query, command, callbacks, context, cursorPosition) {
  const fuseCommand = command;
  const innerText = query.substring(0, cursorPosition);
  const position = getPosition(innerText, fuseCommand);
  switch (position) {
    // FUSE arguments can be suggested in any order, except for <fuse_method> which should come first if present.
    // <fuse_method>, SCORE BY, KEY BY, GROUP BY, WITH
    case FusePosition.BEFORE_NEW_ARGUMENT:
      return fuseArgumentsAutocomplete(fuseCommand);

    // SCORE BY suggests a single field of double type
    case FusePosition.SCORE_BY:
      return await scoreByAutocomplete(innerText, callbacks, context);

    // GROUP BY suggests a single field of string type
    case FusePosition.GROUP_BY:
      return await groupByAutocomplete(innerText, callbacks, context);

    // KEY BY suggests multiple fields of string type
    case FusePosition.KEY_BY:
      return await keyByAutocomplete(innerText, fuseCommand, callbacks, context);

    // WITH suggests a map of options that depends on the <fuse_method>
    case FusePosition.WITH:
      return await withOptionAutocomplete(innerText, fuseCommand);
  }
}

/**
 * Returns suggestions for the `SCORE BY` argument of the `FUSE` command.
 * Returns fields of double type.
 */
async function scoreByAutocomplete(innerText, callbacks, context) {
  var _callbacks$getByType;
  const numericFields = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getByType = callbacks.getByType) === null || _callbacks$getByType === void 0 ? void 0 : _callbacks$getByType.call(callbacks, 'double', [], {
    advanceCursor: true,
    openSuggestions: true
  }));
  const isFragmentComplete = fragment => (0, _helpers.columnExists)(fragment, context);
  const getSuggestionsForIncomplete = (_fragment, rangeToReplace) => {
    var _numericFields$map;
    return (_numericFields$map = numericFields === null || numericFields === void 0 ? void 0 : numericFields.map(suggestion => {
      return {
        ...suggestion,
        rangeToReplace
      };
    })) !== null && _numericFields$map !== void 0 ? _numericFields$map : [];
  };
  const getSuggestionsForComplete = () => [];
  return await (0, _helpers.handleFragment)(innerText, isFragmentComplete, getSuggestionsForIncomplete, getSuggestionsForComplete);
}

/**
 *  Returns suggestions for the `GROUP BY` argument of the `FUSE` command.
 *  Returns fields of string type.
 */
async function groupByAutocomplete(innerText, callbacks, context) {
  var _callbacks$getByType2;
  const stringFields = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getByType2 = callbacks.getByType) === null || _callbacks$getByType2 === void 0 ? void 0 : _callbacks$getByType2.call(callbacks, _types.ESQL_STRING_TYPES, [], {
    advanceCursor: true,
    openSuggestions: true
  }));
  const isFragmentComplete = fragment => (0, _helpers.columnExists)(fragment, context);
  const getSuggestionsForIncomplete = (_fragment, rangeToReplace) => {
    var _stringFields$map;
    return (_stringFields$map = stringFields === null || stringFields === void 0 ? void 0 : stringFields.map(suggestion => {
      return {
        ...suggestion,
        rangeToReplace
      };
    })) !== null && _stringFields$map !== void 0 ? _stringFields$map : [];
  };
  const getSuggestionsForComplete = () => [];
  return await (0, _helpers.handleFragment)(innerText, isFragmentComplete, getSuggestionsForIncomplete, getSuggestionsForComplete);
}

/**
 * Returns suggestions for the `KEY BY` argument of the `FUSE` command.
 * Returns fields of string type that are not already used in the `KEY BY` argument.
 * If there are already fields used, it also suggests a comma to add another field or
 *  other FUSE arguments configurations to scape from KEY BY.
 */
async function keyByAutocomplete(innerText, command, callbacks, context) {
  var _keyByOption$args$map, _await$callbacks$getB, _callbacks$getByType3;
  const keyByOption = (0, _utils.findCommandOptionByName)(command, 'key by');
  const alreadyUsedFields = (_keyByOption$args$map = keyByOption === null || keyByOption === void 0 ? void 0 : keyByOption.args.map(arg => (0, _is.isColumn)(arg) ? arg.name : '')) !== null && _keyByOption$args$map !== void 0 ? _keyByOption$args$map : [];
  const allFields = (_await$callbacks$getB = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getByType3 = callbacks.getByType) === null || _callbacks$getByType3 === void 0 ? void 0 : _callbacks$getByType3.call(callbacks, _types.ESQL_STRING_TYPES, alreadyUsedFields, {
    openSuggestions: true
  }))) !== null && _await$callbacks$getB !== void 0 ? _await$callbacks$getB : [];
  const isFragmentComplete = fragment => (0, _helpers.columnExists)(fragment, context);
  const getSuggestionsForComplete = (fragment, rangeToReplace) => {
    const finalSuggestions = fuseArgumentsAutocomplete(command).map(s => ({
      ...s,
      text: ` ${s.text}`
    }));
    if (allFields.length > 0) {
      finalSuggestions.push({
        ..._.commaCompleteItem,
        text: ', '
      });
    }
    return finalSuggestions.map(s => (0, _.withAutoSuggest)({
      ...s,
      filterText: fragment,
      text: fragment + s.text,
      rangeToReplace
    }));
  };
  const getSuggestionsForIncomplete = (_fragment, rangeToReplace) => {
    return allFields.map(suggestion => (0, _.withAutoSuggest)({
      ...suggestion,
      rangeToReplace
    }));
  };
  return (0, _helpers.handleFragment)(innerText, isFragmentComplete, getSuggestionsForIncomplete, getSuggestionsForComplete);
}

/**
 * Returns suggestions for the `WITH` argument of the `FUSE` command.
 * The suggestions depend on the `<fuse_method>` used.
 * The default fuse method is `rrf`.
 * If `rrf`, it suggests `rank_constant` and `weights`.
 * If `linear` method is used, it suggests `normalizer` and `weights`.
 */
async function withOptionAutocomplete(innerText, command) {
  const withOption = (0, _utils.findCommandOptionByName)(command, 'with');
  if (!withOption) {
    return [];
  }

  // Means the map is not present - FUSE WITH /
  if (withOption.args.length === 0) {
    return [(0, _.withAutoSuggest)({
      label: '{ }',
      kind: 'Reference',
      detail: '{ ... }',
      text: '{ $0 }',
      sortText: '0',
      asSnippet: true
    })];
  }

  // Select map parameters based on fuseType; default to rrf if missing, if unknown keep the empty map and don't suggest nothing
  let mapParameters = {};
  if (!command.fuseType || command.fuseType.text === 'rrf') {
    mapParameters = {
      rank_constant: {
        type: 'number',
        suggestions: [{
          label: '60',
          text: '60',
          kind: 'Value',
          sortText: '1',
          detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.fuse.rank_constant_default', {
            defaultMessage: 'Default value'
          })
        }]
      },
      weights: {
        type: 'map'
      }
    };
  } else if (command.fuseType.text === 'linear') {
    mapParameters = {
      normalizer: {
        type: 'string',
        suggestions: [_.noneValueCompleteItem, _.minMaxValueCompleteItem]
      },
      weights: {
        type: 'map'
      }
    };
  }
  return (0, _map_expression.getCommandMapExpressionSuggestions)(innerText, mapParameters);
}

/**
 * Returns suggestions for the arguments of the `FUSE` command.
 * Arguments can be placed at any order, except for the `<fuse_method>` which should be the first argument if present.
 * These arguments are: `<fuse_method>`, `SCORE BY`, `GROUP BY`, `KEY BY`, `WITH`.
 * It also suggests a pipe to chain another command after the `FUSE` command.
 */
function fuseArgumentsAutocomplete(command) {
  const suggestions = [_.pipeCompleteItem];
  const {
    scoreBy,
    keyBy,
    groupBy,
    withOption
  } = (0, _utils.extractFuseArgs)(command);
  if (command.args.length === 0) {
    suggestions.push({
      label: 'linear',
      kind: 'Value',
      detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.fuse.linear', {
        defaultMessage: 'Linear combination of scores'
      }),
      text: 'linear ',
      sortText: '0'
    }, {
      label: 'rrf',
      kind: 'Value',
      detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.fuse.rrf', {
        defaultMessage: 'Reciprocal rank fusion'
      }),
      text: 'rrf ',
      sortText: '0'
    });
  }
  if (!scoreBy) {
    suggestions.push({
      label: 'SCORE BY',
      kind: 'Reference',
      detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.fuse.scoreBy', {
        defaultMessage: 'Defaults to _score. Designates which column to use to retrieve the relevance scores of the input'
      }),
      text: 'SCORE BY ',
      sortText: '1'
    });
  }
  if (!groupBy) {
    suggestions.push({
      label: 'GROUP BY',
      kind: 'Reference',
      detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.fuse.groupBy', {
        defaultMessage: 'Defaults to _fork. Designates which column represents the result set'
      }),
      text: 'GROUP BY ',
      sortText: '2'
    });
  }
  if (!keyBy) {
    suggestions.push({
      label: 'KEY BY',
      kind: 'Reference',
      detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.fuse.keyBy', {
        defaultMessage: 'Defaults to _id, _index. Rows with matching key_columns values are merged'
      }),
      text: 'KEY BY ',
      sortText: '3'
    });
  }
  if (!withOption) {
    suggestions.push({
      ..._.withCompleteItem,
      sortText: '4'
    });
  }
  return suggestions.map(s => (0, _.withAutoSuggest)(s));
}