"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStreams = getDataStreams;
exports.getDatasetTypesPrivileges = getDatasetTypesPrivileges;
var _constants = require("../../../../common/constants");
var _utils = require("../../../../common/utils");
var _services = require("../../../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getDataStreams(options) {
  const {
    esClient,
    types = [],
    datasetQuery,
    uncategorisedOnly
  } = options;
  const datasetNames = datasetQuery ? [datasetQuery] : types.map(type => (0, _utils.streamPartsToIndexPattern)({
    typePattern: type,
    datasetPattern: '*-*'
  }));
  const datasetUserPrivileges = await _services.datasetQualityPrivileges.getDatasetPrivileges(esClient, datasetNames);
  const canMonitor = Object.values(datasetUserPrivileges.datasetsPrivilages).some(privileges => privileges.canMonitor);
  if (!canMonitor) {
    return {
      dataStreams: [],
      datasetUserPrivileges
    };
  }
  const allDataStreams = await _services.dataStreamService.getMatchingDataStreams(esClient, datasetNames.join(','));
  const filteredDataStreams = uncategorisedOnly ? allDataStreams.filter(stream => {
    return !stream._meta || !stream._meta.managed_by || stream._meta.managed_by !== 'fleet';
  }) : allDataStreams;
  const dataStreamsPrivileges = filteredDataStreams.length ? await _services.datasetQualityPrivileges.getHasIndexPrivileges(esClient, filteredDataStreams.map(({
    name
  }) => name), ['monitor', _constants.FAILURE_STORE_PRIVILEGE, _constants.MANAGE_FAILURE_STORE_PRIVILEGE]) : {};
  const mappedDataStreams = filteredDataStreams.map(dataStream => {
    var _dataStream$failure_s, _dataStream$_meta, _dataStream$_meta$pac, _dataStreamsPrivilege, _dataStreamsPrivilege2, _dataStreamsPrivilege3, _dataStream$failure_s2;
    // @ts-expect-error
    const failureStoreLifecycle = (_dataStream$failure_s = dataStream.failure_store) === null || _dataStream$failure_s === void 0 ? void 0 : _dataStream$failure_s.lifecycle;
    return {
      name: dataStream.name,
      integration: (_dataStream$_meta = dataStream._meta) === null || _dataStream$_meta === void 0 ? void 0 : (_dataStream$_meta$pac = _dataStream$_meta.package) === null || _dataStream$_meta$pac === void 0 ? void 0 : _dataStream$_meta$pac.name,
      // @ts-expect-error
      lastActivity: dataStream.maximum_timestamp,
      userPrivileges: {
        canMonitor: (_dataStreamsPrivilege = dataStreamsPrivileges[dataStream.name]) === null || _dataStreamsPrivilege === void 0 ? void 0 : _dataStreamsPrivilege.monitor,
        canReadFailureStore: (_dataStreamsPrivilege2 = dataStreamsPrivileges[dataStream.name]) === null || _dataStreamsPrivilege2 === void 0 ? void 0 : _dataStreamsPrivilege2[_constants.FAILURE_STORE_PRIVILEGE],
        canManageFailureStore: (_dataStreamsPrivilege3 = dataStreamsPrivileges[dataStream.name]) === null || _dataStreamsPrivilege3 === void 0 ? void 0 : _dataStreamsPrivilege3[_constants.MANAGE_FAILURE_STORE_PRIVILEGE]
      },
      hasFailureStore: (_dataStream$failure_s2 = dataStream.failure_store) === null || _dataStream$failure_s2 === void 0 ? void 0 : _dataStream$failure_s2.enabled,
      customRetentionPeriod: failureStoreLifecycle === null || failureStoreLifecycle === void 0 ? void 0 : failureStoreLifecycle.data_retention,
      defaultRetentionPeriod: (failureStoreLifecycle === null || failureStoreLifecycle === void 0 ? void 0 : failureStoreLifecycle.retention_determined_by) === 'default_failures_retention' ? failureStoreLifecycle === null || failureStoreLifecycle === void 0 ? void 0 : failureStoreLifecycle.effective_retention : undefined
    };
  });
  return {
    dataStreams: mappedDataStreams,
    datasetUserPrivileges
  };
}
async function getDatasetTypesPrivileges(options) {
  const {
    esClient,
    types
  } = options;
  const datasetNames = types.map(type => (0, _utils.streamPartsToIndexPattern)({
    typePattern: type,
    datasetPattern: '*-*'
  }));
  const {
    datasetsPrivilages
  } = await _services.datasetQualityPrivileges.getDatasetPrivileges(esClient, datasetNames);
  return {
    datasetsPrivilages
  };
}