"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getVirtualVersionMap = exports.getModelVersionMapForTypes = exports.getLatestModelVersion = exports.getLatestMigrationVersion = exports.getLatestMappingsVirtualVersionMap = exports.getLatestMappingsVersionNumber = exports.getLatestMappingsModelVersion = exports.getCurrentVirtualVersion = void 0;
var _semver = require("semver");
var _conversion = require("./conversion");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Represents the virtual version of a given SO type.
 * The virtual version is a compatibility format between the old
 * migration system's versioning, based on the stack version, and the new model versioning.
 *
 * A virtual version is a plain semver version. Depending on its major version value, the
 * underlying version can be the following:
 * - Major < 10: Old migrations system (stack versions), using the equivalent value (e.g `8.7.0` => migration version `8.7.0`)
 * - Major == 10: Model versions, using the `10.{modelVersion}.0` format (e.g `10.3.0` => model version 3)
 */

/**
 * A map of SO type name to Model Version.
 */

/**
 * A map of SO type name to {@link VirtualVersion}.
 */

/**
 * Returns the latest registered model version number for the given type.
 */
const getLatestModelVersion = type => {
  var _type$modelVersions;
  const versionMap = typeof type.modelVersions === 'function' ? type.modelVersions() : (_type$modelVersions = type.modelVersions) !== null && _type$modelVersions !== void 0 ? _type$modelVersions : {};
  return Object.keys(versionMap).reduce((memo, current) => {
    return Math.max(memo, (0, _conversion.assertValidModelVersion)(current));
  }, 0);
};

/**
 * Returns the latest registered migration version for the given type.
 * @param type the saved object from which we want to retrieve the latest migration version
 * @returns string the latest migration appearing in the migrations property, or 0.0.0 if no migrations defined.
 */
exports.getLatestModelVersion = getLatestModelVersion;
const getLatestMigrationVersion = type => {
  var _type$migrations;
  const migrationMap = typeof type.migrations === 'function' ? type.migrations() : (_type$migrations = type.migrations) !== null && _type$migrations !== void 0 ? _type$migrations : {};
  return Object.keys(migrationMap).reduce((memo, current) => {
    return (0, _semver.gt)(memo, current) ? memo : current;
  }, '0.0.0');
};

/**
 * Build a version map for the given types.
 */
exports.getLatestMigrationVersion = getLatestMigrationVersion;
const getModelVersionMapForTypes = types => {
  return types.reduce((versionMap, type) => {
    versionMap[type.name] = getLatestModelVersion(type);
    return versionMap;
  }, {});
};

/**
 * Returns the current virtual version for the given type.
 * It will either be:
 * - if the type defines model versions => the latest model version
 * - if the type does NOT define model versions =>
 *   - the initialModelVersion aka 10.0.0 (if useModelVersionsOnly is set to true)
 *   - the latest migration version for the type (if useModelVersionsOnly is set to false)
 */
exports.getModelVersionMapForTypes = getModelVersionMapForTypes;
const getCurrentVirtualVersion = (type, useModelVersionsOnly) => {
  if (type.modelVersions || useModelVersionsOnly) {
    const versionNumber = getLatestModelVersion(type);
    return (0, _conversion.modelVersionToVirtualVersion)(versionNumber);
  } else {
    return getLatestMigrationVersion(type);
  }
};
exports.getCurrentVirtualVersion = getCurrentVirtualVersion;
/**
 * Returns a map of virtual model version for the given types.
 * See {@link getCurrentVirtualVersion}
 */
const getVirtualVersionMap = ({
  types,
  useModelVersionsOnly
}) => {
  return types.reduce((versionMap, type) => {
    versionMap[type.name] = getCurrentVirtualVersion(type, useModelVersionsOnly);
    return versionMap;
  }, {});
};

/**
 * Returns the latest version number that includes changes in the mappings, for the given type.
 * If none of the versions are updating the mappings, it will return 0
 */
exports.getVirtualVersionMap = getVirtualVersionMap;
const getLatestMappingsVersionNumber = type => {
  var _type$modelVersions2;
  const versionMap = typeof type.modelVersions === 'function' ? type.modelVersions() : (_type$modelVersions2 = type.modelVersions) !== null && _type$modelVersions2 !== void 0 ? _type$modelVersions2 : {};
  return Object.entries(versionMap).filter(([version, info]) => {
    var _info$changes;
    return (_info$changes = info.changes) === null || _info$changes === void 0 ? void 0 : _info$changes.some(change => change.type === 'mappings_addition');
  }).reduce((memo, [current]) => {
    return Math.max(memo, (0, _conversion.assertValidModelVersion)(current));
  }, 0);
};

/**
 * Returns the latest model version that includes changes in the mappings, for the given type.
 * It will either be a model version or the latest migration version
 * if no changed were introduced after enforcing the switch to model versions.
 */
exports.getLatestMappingsVersionNumber = getLatestMappingsVersionNumber;
const getLatestMappingsModelVersion = type => {
  const modelVersion = getLatestMappingsVersionNumber(type);
  return (0, _conversion.modelVersionToVirtualVersion)(modelVersion);
};

/**
 * Returns a map of virtual model version for the given types.
 * See {@link getLatestMappingsModelVersion}
 */
exports.getLatestMappingsModelVersion = getLatestMappingsModelVersion;
const getLatestMappingsVirtualVersionMap = types => {
  return types.reduce((versionMap, type) => {
    versionMap[type.name] = getLatestMappingsModelVersion(type);
    return versionMap;
  }, {});
};
exports.getLatestMappingsVirtualVersionMap = getLatestMappingsVirtualVersionMap;