"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EndpointExceptionsValidator = void 0;
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _endpoint_exception_errors = require("./endpoint_exception_errors");
var _base_validator = require("./base_validator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EndpointExceptionsValidator extends _base_validator.BaseValidator {
  static isEndpointException(item) {
    return item.listId === _securitysolutionListConstants.ENDPOINT_LIST_ID;
  }
  async validateHasReadPrivilege() {
    return this.validateHasEndpointExceptionsPrivileges('canReadEndpointExceptions');
  }
  async validateHasWritePrivilege() {
    await this.validateHasEndpointExceptionsPrivileges('canWriteEndpointExceptions');

    // Endpoint Exceptions are currently ONLY global, so we need to make sure the user
    // also has the new Global Artifacts privilege
    try {
      await this.validateHasPrivilege('canManageGlobalArtifacts');
    } catch (error) {
      // We provide a more detailed error here
      throw new _endpoint_exception_errors.EndpointExceptionsValidationError(`${error.message}. ${_base_validator.GLOBAL_ARTIFACT_MANAGEMENT_NOT_ALLOWED_MESSAGE}`, 403);
    }
  }
  async validatePreCreateItem(item) {
    await this.validateHasWritePrivilege();
    await this.validateCanCreateGlobalArtifacts(item);
    await this.validateCreateOwnerSpaceIds(item);
    return item;
  }
  async validatePreUpdateItem(item, currentItem) {
    await this.validateHasWritePrivilege();
    await this.validateUpdateOwnerSpaceIds(item, currentItem);
    await this.validateCanUpdateItemInActiveSpace(item, currentItem);
    return item;
  }
  async validatePreDeleteItem(currentItem) {
    await this.validateHasWritePrivilege();
    await this.validateCanDeleteItemInActiveSpace(currentItem);
  }
  async validatePreGetOneItem(currentItem) {
    await this.validateHasReadPrivilege();
    await this.validateCanReadItemInActiveSpace(currentItem);
  }
  async validatePreMultiListFind() {
    await this.validateHasReadPrivilege();
  }
  async validatePreExport() {
    await this.validateHasReadPrivilege();
  }
  async validatePreSingleListFind() {
    await this.validateHasReadPrivilege();
  }
  async validatePreGetListSummary() {
    await this.validateHasReadPrivilege();
  }
}
exports.EndpointExceptionsValidator = EndpointExceptionsValidator;