"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GLOBAL_ARTIFACT_MANAGEMENT_NOT_ALLOWED_MESSAGE = exports.BasicEndpointExceptionDataSchema = exports.BaseValidator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _configSchema = require("@kbn/config-schema");
var _fp = require("lodash/fp");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
var _i18n = require("@kbn/i18n");
require("@kbn/lists-plugin/server/services/exception_lists/exception_list_client_types");
var _lodash = require("lodash");
var _stringify = require("../../../endpoint/utils/stringify");
var _errors = require("../../../endpoint/errors");
var _utils = require("../../../../common/endpoint/service/artifacts/utils");
var _authz = require("../../../../common/endpoint/service/authz");
var _artifacts = require("../../../../common/endpoint/service/artifacts");
var _errors2 = require("./errors");
var _endpoint_exception_errors = require("./endpoint_exception_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OWNER_SPACE_ID_TAG_MANAGEMENT_NOT_ALLOWED_MESSAGE = _i18n.i18n.translate('xpack.securitySolution.baseValidator.noGlobalArtifactAuthzApiMessage', {
  defaultMessage: 'Management of "ownerSpaceId" tag requires global artifact management privilege'
});
const GLOBAL_ARTIFACT_MANAGEMENT_NOT_ALLOWED_MESSAGE = exports.GLOBAL_ARTIFACT_MANAGEMENT_NOT_ALLOWED_MESSAGE = _i18n.i18n.translate('xpack.securitySolution.baseValidator.noGlobalArtifactManagementMessage', {
  defaultMessage: 'Management of global artifacts requires additional privilege (global artifact management)'
});
const ITEM_CANNOT_BE_MANAGED_IN_CURRENT_SPACE_MESSAGE = spaceIds => _i18n.i18n.translate('xpack.securitySolution.baseValidator.cannotManageItemInCurrentSpace', {
  defaultMessage: `Updates to this shared item can only be done from the following space {numberOfSpaces, plural, one {ID} other {IDs} }: {itemOwnerSpaces} (or by someone having global artifact management privilege)`,
  values: {
    numberOfSpaces: spaceIds.length,
    itemOwnerSpaces: spaceIds.join(', ')
  }
});
const BasicEndpointExceptionDataSchema = exports.BasicEndpointExceptionDataSchema = _configSchema.schema.object({
  // must have a name
  name: _configSchema.schema.string({
    minLength: 1,
    maxLength: 256
  }),
  description: _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 0,
    maxLength: 256,
    defaultValue: ''
  })),
  // We only support agnostic entries
  namespaceType: _configSchema.schema.literal('agnostic'),
  // only one OS per entry
  osTypes: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.literal(_securitysolutionUtils.OperatingSystem.WINDOWS), _configSchema.schema.literal(_securitysolutionUtils.OperatingSystem.LINUX), _configSchema.schema.literal(_securitysolutionUtils.OperatingSystem.MAC)]), {
    minSize: 1,
    maxSize: 1
  })
},
// Because we are only validating some fields from the Exception Item, we set `unknowns` to `ignore` here
{
  unknowns: 'ignore'
});

/**
 * Provides base methods for doing validation that apply across endpoint exception entries
 */
class BaseValidator {
  constructor(endpointAppContext,
  /**
   * Request is optional only because it needs to be optional in the Lists ExceptionListClient
   */
  request) {
    var _this$constructor$nam;
    (0, _defineProperty2.default)(this, "endpointAuthzPromise", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.endpointAppContext = endpointAppContext;
    this.request = request;
    this.logger = endpointAppContext.createLogger((_this$constructor$nam = this.constructor.name) !== null && _this$constructor$nam !== void 0 ? _this$constructor$nam : 'artifactBaseValidator');
    if (this.request) {
      this.endpointAuthzPromise = this.endpointAppContext.getEndpointAuthz(this.request);
    } else {
      this.endpointAuthzPromise = Promise.resolve((0, _authz.getEndpointAuthzInitialState)());
    }
  }
  notifyFeatureUsage(item, featureKey) {
    if (this.isItemByPolicy(item) && featureKey.endsWith('_BY_POLICY') || !this.isItemByPolicy(item) && !featureKey.endsWith('_BY_POLICY')) {
      this.endpointAppContext.getFeatureUsageService().notifyUsage(featureKey);
    }
  }
  async validateHasEndpointExceptionsPrivileges(privilege) {
    if (!(await this.endpointAuthzPromise)[privilege]) {
      throw new _endpoint_exception_errors.EndpointExceptionsValidationError('Endpoint exceptions authorization failure', 403);
    }
  }
  async validateHasPrivilege(privilege) {
    if (!(await this.endpointAuthzPromise)[privilege]) {
      throw new _errors2.EndpointArtifactExceptionValidationError(_errors.ENDPOINT_AUTHZ_ERROR_MESSAGE, 403);
    }
  }
  isItemByPolicy(item) {
    return (0, _artifacts.isArtifactByPolicy)(item);
  }
  async isAllowedToCreateArtifactsByPolicy() {
    return (await this.endpointAuthzPromise).canCreateArtifactsByPolicy;
  }
  async validateCanManageEndpointArtifacts() {
    if (!(await this.endpointAuthzPromise).canAccessEndpointManagement) {
      throw new _errors2.EndpointArtifactExceptionValidationError(_errors.ENDPOINT_AUTHZ_ERROR_MESSAGE, 403);
    }
  }
  async validateCanIsolateHosts() {
    if (!(await this.endpointAuthzPromise).canIsolateHost) {
      throw new _errors2.EndpointArtifactExceptionValidationError(_errors.ENDPOINT_AUTHZ_ERROR_MESSAGE, 403);
    }
  }

  /**
   * validates some basic common data that can be found across all endpoint exceptions
   * @param item
   * @protected
   */
  async validateBasicData(item) {
    try {
      BasicEndpointExceptionDataSchema.validate(item);
    } catch (error) {
      throw new _errors2.EndpointArtifactExceptionValidationError(error.message);
    }
  }
  async validateCanCreateByPolicyArtifacts(item) {
    if (this.isItemByPolicy(item) && !(await this.isAllowedToCreateArtifactsByPolicy())) {
      throw new _errors2.EndpointArtifactExceptionValidationError('Your license level does not allow create/update of by policy artifacts', 403);
    }
  }

  /**
   * Validates that by-policy artifacts is permitted and that each policy referenced in the item is valid
   * @protected
   */
  async validateByPolicyItem(item, /** Should be provided when an existing item is being updated. Will `undefined` on create flows */
  currentItem) {
    if (this.isItemByPolicy(item)) {
      var _await$packagePolicy$;
      const spaceId = await this.getActiveSpaceId();
      const {
        packagePolicy,
        savedObjects
      } = this.endpointAppContext.getInternalFleetServices(spaceId);
      const policyIds = (0, _artifacts.getPolicyIdsFromArtifact)(item);
      const soClient = savedObjects.createInternalScopedSoClient({
        spaceId
      });
      if (policyIds.length === 0) {
        return;
      }
      const policiesFromFleet = (_await$packagePolicy$ = await packagePolicy.getByIDs(soClient, policyIds, {
        ignoreMissing: true
      })) !== null && _await$packagePolicy$ !== void 0 ? _await$packagePolicy$ : [];
      this.logger.debug(() => `Lookup of policy ids:\n[${policyIds.join(' | ')}] for space [${spaceId}] returned:\n${(0, _stringify.stringify)(policiesFromFleet.map(policy => ({
        id: policy.id,
        name: policy.name,
        spaceIds: policy.spaceIds
      })))}`);
      let invalidPolicyIds = policyIds.filter(policyId => !policiesFromFleet.some(policy => policyId === policy.id));
      if (invalidPolicyIds.length > 0 && currentItem) {
        const currentItemPolicyIds = (0, _artifacts.getPolicyIdsFromArtifact)(currentItem);

        // Check to see if the invalid policy IDs are ones that the current item (pre-update) already has,
        // which implies that they are valid, but not visible in the active space.
        invalidPolicyIds = invalidPolicyIds.filter(id => !currentItemPolicyIds.includes(id));
      }
      if (invalidPolicyIds.length) {
        throw new _errors2.EndpointArtifactExceptionValidationError(`invalid policy ids: ${invalidPolicyIds.join(', ')}`);
      }
    }
  }

  /**
   * If the item being updated is `by policy`, method validates if anything was changes in regard to
   * the effected scope of the by policy settings.
   *
   * @param updatedItem
   * @param currentItem
   * @protected
   */
  wasByPolicyEffectScopeChanged(updatedItem, currentItem) {
    // if global, then return. Nothing to validate and setting the trusted app to global is allowed
    if (!this.isItemByPolicy(updatedItem)) {
      return false;
    }
    if (updatedItem.tags) {
      return !(0, _fp.isEqual)((0, _artifacts.getPolicyIdsFromArtifact)({
        tags: updatedItem.tags
      }), (0, _artifacts.getPolicyIdsFromArtifact)(currentItem));
    }
    return false;
  }
  async validateUpdateOwnerSpaceIds(updatedItem, currentItem) {
    if (this.wasOwnerSpaceIdTagsChanged(updatedItem, currentItem) && !(await this.endpointAuthzPromise).canManageGlobalArtifacts) {
      throw new _errors2.EndpointArtifactExceptionValidationError(`${_errors.ENDPOINT_AUTHZ_ERROR_MESSAGE}. ${OWNER_SPACE_ID_TAG_MANAGEMENT_NOT_ALLOWED_MESSAGE}`, 403);
    }
  }
  async validateCreateOwnerSpaceIds(item) {
    if (item.tags && item.tags.length > 0) {
      if ((await this.endpointAuthzPromise).canManageGlobalArtifacts) {
        return;
      }
      const ownerSpaceIds = (0, _utils.getArtifactOwnerSpaceIds)(item);
      const activeSpaceId = await this.getActiveSpaceId();
      if (ownerSpaceIds.length > 1 || ownerSpaceIds.length === 1 && ownerSpaceIds[0] !== activeSpaceId) {
        throw new _errors2.EndpointArtifactExceptionValidationError(`${_errors.ENDPOINT_AUTHZ_ERROR_MESSAGE}. ${OWNER_SPACE_ID_TAG_MANAGEMENT_NOT_ALLOWED_MESSAGE}`, 403);
      }
    }
  }
  wasOwnerSpaceIdTagsChanged(updatedItem, currentItem) {
    return !(0, _fp.isEqual)((0, _utils.getArtifactOwnerSpaceIds)(updatedItem), (0, _utils.getArtifactOwnerSpaceIds)(currentItem));
  }
  async getActiveSpaceId() {
    if (!this.request) {
      throw new _errors2.EndpointArtifactExceptionValidationError('Unable to determine space id. Missing HTTP Request object', 500);
    }
    return (await this.endpointAppContext.getActiveSpace(this.request)).id;
  }
  async validateCanCreateGlobalArtifacts(item) {
    if (!this.isItemByPolicy(item) && !(await this.endpointAuthzPromise).canManageGlobalArtifacts) {
      throw new _errors2.EndpointArtifactExceptionValidationError(`${_errors.ENDPOINT_AUTHZ_ERROR_MESSAGE}. ${GLOBAL_ARTIFACT_MANAGEMENT_NOT_ALLOWED_MESSAGE}`, 403);
    }
  }
  async validateCanUpdateItemInActiveSpace(updatedItem, currentSavedItem) {
    // Those with global artifact management privilege can do it all
    if ((await this.endpointAuthzPromise).canManageGlobalArtifacts) {
      return;
    }

    // If either the updated item or the saved item is a global artifact, then
    // error out - user needs global artifact management privilege
    if (!this.isItemByPolicy(updatedItem) || !this.isItemByPolicy(currentSavedItem)) {
      throw new _errors2.EndpointArtifactExceptionValidationError(`${_errors.ENDPOINT_AUTHZ_ERROR_MESSAGE}. ${GLOBAL_ARTIFACT_MANAGEMENT_NOT_ALLOWED_MESSAGE}`, 403);
    }
    const itemOwnerSpaces = (0, _utils.getArtifactOwnerSpaceIds)(currentSavedItem);

    // Per-space items can only be managed from one of the `ownerSpaceId`'s
    if (!itemOwnerSpaces.includes(await this.getActiveSpaceId())) {
      throw new _errors2.EndpointArtifactExceptionValidationError(ITEM_CANNOT_BE_MANAGED_IN_CURRENT_SPACE_MESSAGE(itemOwnerSpaces), 403);
    }
  }
  async validateCanDeleteItemInActiveSpace(currentSavedItem) {
    // Those with global artifact management privilege can do it all
    if ((await this.endpointAuthzPromise).canManageGlobalArtifacts) {
      return;
    }

    // If item is a global artifact then error - user must have global artifact management privilege
    if (!this.isItemByPolicy(currentSavedItem)) {
      throw new _errors2.EndpointArtifactExceptionValidationError(`${_errors.ENDPOINT_AUTHZ_ERROR_MESSAGE}. ${GLOBAL_ARTIFACT_MANAGEMENT_NOT_ALLOWED_MESSAGE}`, 403);
    }
    const itemOwnerSpaces = (0, _utils.getArtifactOwnerSpaceIds)(currentSavedItem);

    // Per-space items can only be deleted from one of the `ownerSpaceId`'s
    if (!itemOwnerSpaces.includes(await this.getActiveSpaceId())) {
      throw new _errors2.EndpointArtifactExceptionValidationError(ITEM_CANNOT_BE_MANAGED_IN_CURRENT_SPACE_MESSAGE(itemOwnerSpaces), 403);
    }
  }
  async validateCanReadItemInActiveSpace(currentSavedItem) {
    this.logger.debug(() => `Validating if can read single item:\n${(0, _stringify.stringify)(currentSavedItem)}`);

    // Everyone can read global artifacts and those with global artifact management privilege can do it all
    if ((0, _utils.isArtifactGlobal)(currentSavedItem) || (await this.endpointAuthzPromise).canManageGlobalArtifacts) {
      return;
    }
    const activeSpaceId = await this.getActiveSpaceId();
    const ownerSpaceIds = (0, _utils.getArtifactOwnerSpaceIds)(currentSavedItem);
    const policyIds = (0, _artifacts.getPolicyIdsFromArtifact)(currentSavedItem);

    // If per-policy item is not assigned to any policy (dangling artifact) and this artifact
    // is owned by the active space, then allow read.
    if (policyIds.length === 0 && ownerSpaceIds.includes(activeSpaceId)) {
      return;
    }

    // if at least one policy is visible in active space, then allow read
    if (policyIds.length > 0) {
      const {
        packagePolicy,
        savedObjects
      } = this.endpointAppContext.getInternalFleetServices(activeSpaceId);
      const soClient = savedObjects.createInternalScopedSoClient({
        spaceId: activeSpaceId
      });
      const policiesFromFleet = await packagePolicy.getByIDs(soClient, policyIds, {
        ignoreMissing: true
      }).then(packagePolicies => {
        this.logger.debug(() => `Lookup of policy ids:[${policyIds.join(' | ')}]\nvia fleet for space ID [${activeSpaceId}] returned:\n${(0, _stringify.stringify)((packagePolicies !== null && packagePolicies !== void 0 ? packagePolicies : []).map(policy => ({
          id: policy.id,
          name: policy.name,
          spaceIds: policy.spaceIds
        })))}`);
        return (0, _lodash.groupBy)(packagePolicies !== null && packagePolicies !== void 0 ? packagePolicies : [], 'id');
      });
      if (policyIds.some(policyId => Boolean(policiesFromFleet[policyId]))) {
        return;
      }
    }
    this.logger.debug(() => `item can not be read from space [${activeSpaceId}]:\n${(0, _stringify.stringify)(currentSavedItem)}`);
    throw new _endpoint_exception_errors.EndpointExceptionsValidationError(`Item not found in space [${activeSpaceId}]`, 404);
  }
}
exports.BaseValidator = BaseValidator;