"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RssGrowthCircuitBreaker = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Configuration interface for RSS Growth Circuit Breaker.
 */

class RssGrowthCircuitBreaker extends _utils.BaseCircuitBreaker {
  constructor(config) {
    super();
    (0, _defineProperty2.default)(this, "initialRss", void 0);
    (0, _defineProperty2.default)(this, "maxRss", void 0);
    (0, _defineProperty2.default)(this, "maxPercentGrowth", void 0);
    this.config = config;
    if (config.maxRssGrowthPercent < 0 || config.maxRssGrowthPercent > 100) {
      throw new Error('maxRssGrowthPercent must be between 0 and 100');
    }
    this.initialRss = process.memoryUsage().rss;
    this.maxRss = this.initialRss;
    this.maxPercentGrowth = 0;
  }
  async validate() {
    const currentRss = process.memoryUsage().rss;
    const percentGrowth = (currentRss - this.initialRss) / this.initialRss * 100;
    this.maxRss = Math.max(this.maxRss, currentRss);
    this.maxPercentGrowth = Math.max(this.maxPercentGrowth, percentGrowth);
    if (percentGrowth > this.config.maxRssGrowthPercent) {
      return this.failure(`RSS growth exceeded: ${percentGrowth.toFixed(2)}% - max allowed: ${this.config.maxRssGrowthPercent}%`);
    }
    return this.success();
  }
  stats() {
    return {
      initialRss: this.initialRss,
      maxRss: this.maxRss,
      maxPercentGrowth: this.maxPercentGrowth
    };
  }
  validationIntervalMs() {
    return this.config.validationIntervalMs;
  }
}
exports.RssGrowthCircuitBreaker = RssGrowthCircuitBreaker;