"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findOrCreateTagReferences = void 0;
var _find_tags_by_name = require("../../../../tags/saved_objects/find_tags_by_name");
var _create_tag = require("../../../../tags/saved_objects/create_tag");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Finds or creates tags by name and returns tag references for saved objects
 * @param savedObjectsClient - The saved objects client
 * @param tagNames - Array of tag names to find or create
 * @returns Array of tag references that can be used in saved object references
 */
const findOrCreateTagReferences = async (savedObjectsClient, tagNames) => {
  const tagReferences = [];
  for (const tagName of tagNames) {
    let tagResults = await (0, _find_tags_by_name.findTagsByName)({
      savedObjectsClient,
      tagName
    });

    // If tag doesn't exist, create it
    if (tagResults.length === 0) {
      try {
        const createdTag = await (0, _create_tag.createTag)({
          savedObjectsClient,
          tagName,
          description: `Auto-created tag for ${tagName}`
        });
        // Convert SavedObject to SavedObjectsFindResult format
        tagResults = [{
          ...createdTag,
          score: 0
        }];
      } catch (createError) {
        // For now, we'll skip the tag silently
      }
    }
    if (tagResults.length > 0) {
      tagReferences.push({
        id: tagResults[0].id,
        name: `tag-ref-${tagName}`,
        type: 'tag'
      });
    }
  }
  return tagReferences;
};
exports.findOrCreateTagReferences = findOrCreateTagReferences;