"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.searchPrivilegeMonitoringIndicesRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _fp = require("lodash/fp");
var _zodHelpers = require("@kbn/zod-helpers");
var _constants = require("../../../../../common/constants");
var _monitoring = require("../../../../../common/api/entity_analytics/monitoring");
var _assert_advanced_setting_enabled = require("../../utils/assert_advanced_setting_enabled");
var _data_sources_service = require("../data_sources/data_sources_service");
var _saved_object = require("../auth/saved_object");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Return a subset of all indices that contain the user.name field
const LIMIT = 20;
const searchPrivilegeMonitoringIndicesRoute = (router, logger) => {
  router.versioned.get({
    access: 'public',
    path: _constants.PRIVMON_INDICES_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_monitoring.SearchPrivilegesIndicesRequestQuery)
      }
    }
  }, async (context, request, response) => {
    const secSol = await context.securitySolution;
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const query = request.query.searchQuery;
    await (0, _assert_advanced_setting_enabled.assertAdvancedSettingsEnabled)(await context.core, _constants.ENABLE_PRIVILEGED_USER_MONITORING_SETTING);
    const dataClient = secSol.getPrivilegeMonitoringDataClient();
    const config = secSol.getConfig();
    const maxUsersAllowed = config.entityAnalytics.monitoring.privileges.users.maxPrivilegedUsersAllowed;
    const soClient = dataClient.getScopedSoClient(request, {
      includedHiddenTypes: [_saved_object.PrivilegeMonitoringApiKeyType.name, _saved_objects.monitoringEntitySourceType.name]
    });
    const service = (0, _data_sources_service.createDataSourcesService)(dataClient, soClient, maxUsersAllowed);
    try {
      const indices = await service.searchPrivilegesIndices(query);
      return response.ok({
        body: (0, _fp.take)(LIMIT, indices)
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      logger.error(`Error searching privilege monitoring indices: ${error.message}`);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
};
exports.searchPrivilegeMonitoringIndicesRoute = searchPrivilegeMonitoringIndicesRoute;