"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEntityStoreSnapshotTaskState = getEntityStoreSnapshotTaskState;
exports.registerEntityStoreSnapshotTask = registerEntityStoreSnapshotTask;
exports.removeEntityStoreSnapshotTask = removeEntityStoreSnapshotTask;
exports.rewindToYesterday = rewindToYesterday;
exports.runTask = runTask;
exports.startEntityStoreSnapshotTask = startEntityStoreSnapshotTask;
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/core/server");
var _lodash = require("lodash");
var _entity_store = require("../../../../../../common/api/entity_analytics/entity_store");
var _state = require("./state");
var _constants = require("./constants");
var _entity_snapshot_index = require("../../elasticsearch_assets/entity_snapshot_index");
var _utils = require("../../utils");
var _utils2 = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getTaskId(namespace, entityType) {
  return `${_constants.TYPE}:${entityType}:${namespace}:${_constants.VERSION}`;
}
const removeNewlines = content => content.replace(/\n/g, '');
const condenseMultipleSpaces = content => content.replace(/\s+/g, ' ');
const removeComments = content => content.replace(/\/\/.*/g, '');
const minifyPainless = (0, _lodash.flow)(removeComments, removeNewlines, condenseMultipleSpaces);
function registerEntityStoreSnapshotTask({
  logger,
  taskManager,
  getStartServices
}) {
  if (!taskManager) {
    logger.warn('[Entity Store]  Task Manager is unavailable; skipping Entity Store snapshot task registration');
    return;
  }
  const esClientGetter = async () => {
    const [coreStart, _] = await getStartServices();
    return coreStart.elasticsearch.client.asInternalUser;
  };
  taskManager.registerTaskDefinitions({
    [_constants.TYPE]: {
      title: 'Entity Store snapshot task',
      description: `Creates a snapshot every 24h and handles additional data transformations.`,
      timeout: _constants.TIMEOUT,
      maxAttempts: _constants.MAX_ATTEMPTS,
      stateSchemaByVersion: _state.stateSchemaByVersion,
      createTaskRunner: context => {
        return {
          async run() {
            return runTask({
              logger,
              context,
              esClientGetter
            });
          },
          async cancel() {
            logger.warn(`[Entity Store]  Task ${_constants.TYPE} timed out`);
          }
        };
      }
    }
  });
}
async function startEntityStoreSnapshotTask({
  logger,
  namespace,
  entityType,
  taskManager
}) {
  const taskId = getTaskId(namespace, entityType);
  const msg = (0, _utils2.entityStoreTaskLogMessageFactory)(taskId);
  logger.info(msg('attempting to schedule'));
  try {
    const task = await taskManager.ensureScheduled({
      id: taskId,
      taskType: _constants.TYPE,
      scope: _constants.SCOPE,
      schedule: _constants.SCHEDULE,
      state: {
        ..._state.defaultState,
        namespace,
        entityType
      },
      params: {
        version: _constants.VERSION,
        namespace,
        entityType
      }
    });
    logger.info(msg(`scheduled with ${JSON.stringify(task.schedule)}`));
  } catch (e) {
    logger.error(msg(`error scheduling task, received ${e.message}`));
    throw e;
  }
}
async function removeEntityStoreSnapshotTask({
  logger,
  namespace,
  entityType,
  taskManager
}) {
  const taskId = getTaskId(namespace, entityType);
  const msg = (0, _utils2.entityStoreTaskLogMessageFactory)(taskId);
  try {
    await taskManager.remove(getTaskId(namespace, entityType));
    logger.info(msg(`removed snapshot task`));
  } catch (err) {
    if (!_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      logger.error(msg(`failed to remove snapshot task: ${err.message}`));
      throw err;
    }
  }
}

// removeAllFieldsAndResetTimestamp is a painless function that takes a document,
// strips it of all its fields (except for identity fields, like host.name or entity.id),
// and sets the timestamps to @now. The result is used as a script in reindex operation.
const removeAllFieldsAndResetTimestamp = minifyPainless(`
    // Create a new map to hold the filtered fields
    Map newDoc = new HashMap();

    // Keep the entity.id field
    newDoc.entity = new HashMap();
    if (ctx._source.entity?.id != null) {
      newDoc.entity.id = ctx._source.entity.id;
    }
    // Keep host/user/service identity fields if present
    if (ctx._source[params.entityType]?.name != null) {
      newDoc[params.entityType] = new HashMap();
      newDoc[params.entityType].name = ctx._source[params.entityType].name;
    }

    // Set the @timestamp field to the current time
    newDoc['@timestamp'] = params.timestampNow;

    // Set the entity.last_seen_timestamp field to the current time
    newDoc.entity.last_seen_timestamp = params.timestampNow;

    // Reset entity.behaviors fields if they exist
    if (ctx._source.entity?.behaviors != null) {
      if (params.entityType == 'generic') {
        newDoc.entity.behaviors = new HashMap();
        for (String key : ctx._source.entity.behaviors.keySet()) {
          def value = ctx._source.entity.behaviors[key];
          if (value instanceof Boolean || value == 'true') {
            newDoc.entity.behaviors[key] = false;
          }
        }
      } else {
        newDoc[params.entityType].entity = new HashMap();
        newDoc[params.entityType].entity.behaviors = new HashMap();
        for (String key : ctx._source.entity.behaviors.keySet()) {
          def value = ctx._source.entity.behaviors[key];
          if (value instanceof Boolean || value == 'true') {
            newDoc[params.entityType].entity.behaviors[key] = false;
          }
        }
      }
    }

    // Replace the existing document with the new filtered document
    ctx._source = newDoc;
    `);
async function runTask({
  logger,
  context,
  esClientGetter
}) {
  const state = context.taskInstance.state;
  const taskId = context.taskInstance.id;
  const abort = context.abortController;
  const msg = (0, _utils2.entityStoreTaskLogMessageFactory)(taskId);
  const esClient = await esClientGetter();
  try {
    const taskStartTime = (0, _moment.default)().utc();
    const snapshotDate = rewindToYesterday(taskStartTime.toDate());
    logger.info(msg('running task'));
    const entityType = context.taskInstance.params.entityType;
    const namespace = context.taskInstance.params.namespace;
    if (namespace === '') {
      const err = `Task ${taskId} expected vaild namespace in params, got ""`;
      logger.error(msg(err));
      throw err;
    }
    const updatedState = {
      lastExecutionTimestamp: taskStartTime.toISOString(),
      lastSnapshotTookSeconds: 0,
      namespace,
      entityType: entityType,
      runs: state.runs + 1
    };
    if (taskId !== getTaskId(namespace, entityType)) {
      logger.warn(msg('outdated task; exiting'));
      return {
        state: updatedState
      };
    }
    logger.info(msg('creating snapshot index'));
    const {
      index: snapshotIndex
    } = await (0, _entity_snapshot_index.createEntitySnapshotIndex)({
      esClient,
      entityType,
      namespace,
      snapshotDate
    });
    logger.info(msg(`reindexing entities to ${snapshotIndex}`));
    const snapshotReindexResponse = await esClient.reindex({
      source: {
        index: [(0, _utils.getEntitiesIndexName)(entityType, namespace)]
      },
      dest: {
        index: snapshotIndex
      },
      conflicts: 'proceed'
    }, {
      signal: abort.signal
    });
    logger.info(msg(`reindexed to ${snapshotIndex}: ${prettyReindexResponse(snapshotReindexResponse)}`));
    const resetIndex = (0, _utils.getEntitiesResetIndexName)(entityType, namespace);
    logger.info(msg(`removing old entries from ${resetIndex}`));
    const cleanResetResponse = await esClient.deleteByQuery({
      index: resetIndex,
      query: {
        match_all: {}
      },
      refresh: true
    }, {
      signal: abort.signal
    });
    logger.info(msg(`removed ${cleanResetResponse.deleted} old entries from ${resetIndex}`));
    logger.info(msg(`reindexing entities to ${resetIndex}`));
    const resetReindexResponse = await esClient.reindex({
      source: {
        index: [(0, _utils.getEntitiesIndexName)(entityType, namespace)]
      },
      dest: {
        index: resetIndex
      },
      conflicts: 'proceed',
      script: {
        source: removeAllFieldsAndResetTimestamp,
        lang: 'painless',
        params: {
          entityType,
          timestampNow: new Date().toISOString()
        }
      }
    }, {
      signal: abort.signal
    });
    logger.info(msg(`reindexed to ${resetIndex}: ${prettyReindexResponse(resetReindexResponse)}`));
    const taskCompletionTime = (0, _moment.default)().utc().toISOString();
    const taskDurationInSeconds = (0, _moment.default)(taskCompletionTime).diff((0, _moment.default)(taskStartTime), 'seconds');
    updatedState.lastSnapshotTookSeconds = taskDurationInSeconds;
    logger.info(msg(`task run completed in ${taskDurationInSeconds} seconds`));
    return {
      state: updatedState
    };
  } catch (e) {
    logger.error(msg(`error running task, received ${e.message}`));
    throw e;
  }
}
async function getEntityStoreSnapshotTaskState({
  namespace,
  entityType,
  taskManager
}) {
  const taskId = getTaskId(namespace, entityType);
  try {
    const taskState = await taskManager.get(taskId);
    return {
      id: taskState.id,
      resource: _entity_store.EngineComponentResourceEnum.task,
      installed: true,
      enabled: taskState.enabled,
      status: taskState.status,
      retryAttempts: taskState.attempts,
      nextRun: taskState.runAt,
      lastRun: taskState.state.lastExecutionTimestamp,
      runs: taskState.state.runs
    };
  } catch (e) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
      return {
        id: taskId,
        installed: false,
        resource: _entity_store.EngineComponentResourceEnum.task
      };
    }
    throw e;
  }
}

/**
 * Takes a date and returns a date just before midnight the day before. We run
 * snapshot task just after midnight, so we effectively need previous day's
 * date.
 * @param d - The date snapshot task started
 * @returns d - 1 day
 */
function rewindToYesterday(d) {
  return new Date(Date.UTC(d.getUTCFullYear(), d.getUTCMonth(), d.getUTCDate() - 1));
}
function prettyReindexResponse(resp) {
  const {
    created,
    deleted,
    noops,
    failures,
    version_conflicts: versionConflicts,
    retries
  } = resp;
  return JSON.stringify({
    created,
    deleted,
    noops,
    failures,
    versionConflicts,
    retries
  });
}