"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upsertEntity = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _upsert_entity = require("../../../../../../common/api/entity_analytics/entity_store/entities/upsert_entity.gen");
var _constants = require("../../../../../../common/constants");
var _errors = require("../../errors");
var _capability_not_enabled_error = require("../../errors/capability_not_enabled_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const upsertEntity = (router, logger) => {
  router.versioned.put({
    access: 'public',
    path: '/api/entity_store/entities/{entityType}',
    options: {
      availability: {
        stability: 'beta'
      }
    },
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_upsert_entity.UpsertEntityRequestQuery),
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_upsert_entity.UpsertEntityRequestParams),
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_upsert_entity.UpsertEntityRequestBody)
      }
    }
  }, async (context, request, response) => {
    const secSol = await context.securitySolution;
    try {
      await secSol.getEntityStoreCrudClient().upsertEntity(request.params.entityType, request.body, request.query.force);
      return response.ok();
    } catch (error) {
      if (error instanceof _errors.EngineNotRunningError || error instanceof _capability_not_enabled_error.CapabilityNotEnabledError) {
        // Service Unavailable 503
        return response.customError({
          statusCode: 503,
          body: error
        });
      }
      if (error instanceof _errors.BadCRUDRequestError) {
        return response.badRequest({
          body: error
        });
      }
      if (error instanceof _errors.DocumentVersionConflictError) {
        return response.customError({
          statusCode: 409,
          body: error
        });
      }
      logger.error(error);
      throw error;
    }
  });
};
exports.upsertEntity = upsertEntity;