"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteEntity = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _delete_entity = require("../../../../../../common/api/entity_analytics/entity_store/entities/delete_entity.gen");
var _constants = require("../../../../../../common/constants");
var _errors = require("../../errors");
var _capability_not_enabled_error = require("../../errors/capability_not_enabled_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deleteEntity = (router, logger) => {
  router.versioned.delete({
    access: 'public',
    path: '/api/entity_store/entities/{entityType}',
    options: {
      availability: {
        since: '9.3.0',
        stability: 'beta'
      }
    },
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_delete_entity.DeleteSingleEntityRequestParams),
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_delete_entity.DeleteSingleEntityRequestBody)
      }
    }
  }, async (context, request, response) => {
    const secSol = await context.securitySolution;
    try {
      await secSol.getEntityStoreCrudClient().deleteEntity(request.params.entityType, request.body);
      return response.ok({
        body: {
          deleted: true
        }
      });
    } catch (error) {
      if (error instanceof _errors.EngineNotRunningError || error instanceof _capability_not_enabled_error.CapabilityNotEnabledError) {
        // Service Unavailable 503
        return response.customError({
          statusCode: 503,
          body: error
        });
      }
      if (error instanceof _errors.EntityNotFoundError) {
        return response.customError({
          statusCode: 404,
          body: error
        });
      }
      if (error instanceof _errors.BadCRUDRequestError) {
        return response.badRequest({
          body: error
        });
      }
      if (error instanceof _errors.DocumentVersionConflictError) {
        return response.customError({
          statusCode: 409,
          body: error
        });
      }
      if (error instanceof Error) {
        return response.badRequest({
          body: error
        });
      }
      logger.error(error);
      throw error;
    }
  });
};
exports.deleteEntity = deleteEntity;