"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.multiTermsComposite = void 0;
var _pRetry = _interopRequireDefault(require("p-retry"));
var _get_filter = require("../utils/get_filter");
var _single_search_after = require("../utils/single_search_after");
var _build_new_terms_aggregation = require("./build_new_terms_aggregation");
var _utils = require("../utils/utils");
var _build_events_query = require("../utils/build_events_query");
var i18n = _interopRequireWildcard(require("../translations"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * composite aggregation page batch size set to 500 as it shows th best performance(refer https://github.com/elastic/kibana/pull/157413) and
 * allows to be scaled down below when max_clause_count error is encountered
 */
const BATCH_SIZE = 500;
/**
 * This helper does phase2/phase3(look README) got multiple new terms
 * It takes full page of results from phase 1 (10,000)
 * Splits it in chunks (starts from 500) and applies it as a filter in new composite aggregation request
 * It pages through though all 10,000 results from phase1 until maxSize alerts found
 */
const multiTermsCompositeNonRetryable = async ({
  sharedParams,
  filterArgs,
  buckets,
  params,
  aggregatableTimestampField,
  parsedHistoryWindowSize,
  services,
  result,
  logger,
  afterKey,
  createAlertsHook,
  batchSize,
  isAlertSuppressionActive,
  isLoggedRequestsEnabled
}) => {
  const {
    ruleExecutionLogger,
    tuple,
    inputIndex,
    runtimeMappings,
    primaryTimestamp,
    secondaryTimestamp
  } = sharedParams;
  const loggedRequests = [];
  let internalAfterKey = afterKey !== null && afterKey !== void 0 ? afterKey : undefined;
  let i = 0;
  let pageNumber = 0;
  while (i < buckets.length) {
    var _batch;
    pageNumber++;
    const batch = buckets.slice(i, i + batchSize);
    i += batchSize;
    const batchFilters = batch.map(b => {
      const must = Object.keys(b.key).map(key => ({
        match: {
          [key]: b.key[key]
        }
      }));
      return {
        bool: {
          must
        }
      };
    });
    const esFilterForBatch = await (0, _get_filter.getFilter)({
      ...filterArgs,
      filters: [...(Array.isArray(filterArgs.filters) ? filterArgs.filters : []), {
        bool: {
          should: batchFilters
        }
      }]
    });

    // PHASE 2: Take the page of results from Phase 1 and determine if each term exists in the history window.
    // The aggregation filters out buckets for terms that exist prior to `tuple.from`, so the buckets in the
    // response correspond to each new term.
    const searchRequest = (0, _build_events_query.buildEventsSearchQuery)({
      aggregations: (0, _build_new_terms_aggregation.buildCompositeNewTermsAgg)({
        newValueWindowStart: tuple.from,
        timestampField: aggregatableTimestampField,
        fields: params.newTermsFields,
        after: internalAfterKey,
        pageSize: batchSize
      }),
      runtimeMappings,
      searchAfterSortIds: undefined,
      index: inputIndex,
      // For Phase 2, we expand the time range to aggregate over the history window
      // in addition to the rule interval
      from: parsedHistoryWindowSize.toISOString(),
      to: tuple.to.toISOString(),
      filter: esFilterForBatch,
      size: 0,
      primaryTimestamp,
      secondaryTimestamp
    });
    const {
      searchResult: pageSearchResult,
      searchDuration: pageSearchDuration,
      searchErrors: pageSearchErrors,
      loggedRequests: pageSearchLoggedRequests = []
    } = await (0, _single_search_after.singleSearchAfter)({
      searchRequest,
      services,
      ruleExecutionLogger,
      loggedRequestsConfig: isLoggedRequestsEnabled ? {
        type: 'findNewTerms',
        description: i18n.FIND_NEW_TERMS_VALUES_DESCRIPTION((0, _utils.stringifyAfterKey)(internalAfterKey)),
        skipRequestQuery: Boolean(afterKey) || pageNumber > 2
      } : undefined
    });
    result.searchAfterTimes.push(pageSearchDuration);
    result.errors.push(...pageSearchErrors);
    loggedRequests.push(...pageSearchLoggedRequests);
    logger.debug(`Time spent on phase 2 terms agg: ${pageSearchDuration}`);
    const pageSearchResultWithAggs = pageSearchResult;
    if (!pageSearchResultWithAggs.aggregations) {
      throw new Error('Aggregations were missing on new terms search result');
    }

    // PHASE 3: For each term that is not in the history window, fetch the oldest document in
    // the rule interval for that term. This is the first document to contain the new term, and will
    // become the basis of the resulting alert.
    // One document could become multiple alerts if the document contains an array with multiple new terms.
    if (pageSearchResultWithAggs.aggregations.new_terms.buckets.length > 0) {
      const searchRequestPhase3 = (0, _build_events_query.buildEventsSearchQuery)({
        aggregations: (0, _build_new_terms_aggregation.buildCompositeDocFetchAgg)({
          newValueWindowStart: tuple.from,
          timestampField: aggregatableTimestampField,
          fields: params.newTermsFields,
          after: internalAfterKey,
          pageSize: batchSize
        }),
        runtimeMappings,
        searchAfterSortIds: undefined,
        index: inputIndex,
        from: parsedHistoryWindowSize.toISOString(),
        to: tuple.to.toISOString(),
        filter: esFilterForBatch,
        size: 0,
        primaryTimestamp,
        secondaryTimestamp
      });
      const {
        searchResult: docFetchSearchResult,
        searchDuration: docFetchSearchDuration,
        searchErrors: docFetchSearchErrors,
        loggedRequests: docFetchLoggedRequests = []
      } = await (0, _single_search_after.singleSearchAfter)({
        searchRequest: searchRequestPhase3,
        services,
        ruleExecutionLogger,
        loggedRequestsConfig: isLoggedRequestsEnabled ? {
          type: 'findDocuments',
          description: i18n.FIND_NEW_TERMS_EVENTS_DESCRIPTION((0, _utils.stringifyAfterKey)(internalAfterKey)),
          skipRequestQuery: Boolean(afterKey) || pageNumber > 2
        } : undefined
      });
      result.searchAfterTimes.push(docFetchSearchDuration);
      result.errors.push(...docFetchSearchErrors);
      loggedRequests.push(...docFetchLoggedRequests);
      const docFetchResultWithAggs = docFetchSearchResult;
      if (!docFetchResultWithAggs.aggregations) {
        throw new Error('Aggregations were missing on document fetch search result');
      }
      const bulkCreateResult = await createAlertsHook(docFetchResultWithAggs);
      if (bulkCreateResult.alertsWereTruncated) {
        result.warningMessages.push(isAlertSuppressionActive ? (0, _utils.getSuppressionMaxSignalsWarning)() : (0, _utils.getMaxSignalsWarning)());
        return isLoggedRequestsEnabled ? {
          ...bulkCreateResult,
          loggedRequests
        } : bulkCreateResult;
      }
    }
    internalAfterKey = (_batch = batch[batch.length - 1]) === null || _batch === void 0 ? void 0 : _batch.key;
  }
  return {
    loggedRequests
  };
};

/**
 * If request fails with batch size of BATCH_SIZE
 * We will try to reduce it in twice per each request, three times, up until 125
 * Per ES documentation, max_clause_count min value is 1,000 - so with 125 we should be able execute query below max_clause_count value
 */
const multiTermsComposite = async args => {
  let retryBatchSize = BATCH_SIZE;
  const ruleExecutionLogger = args.sharedParams.ruleExecutionLogger;
  return (0, _pRetry.default)(async retryCount => {
    try {
      const res = await multiTermsCompositeNonRetryable({
        ...args,
        batchSize: retryBatchSize
      });
      return res;
    } catch (e) {
      // do not retry if error not related to too many clauses
      // if user's configured rule somehow has filter itself greater than max_clause_count, we won't get to this place anyway,
      // as rule would fail on phase 1
      if (!['query_shard_exception: failed to create query', 'Query contains too many nested clauses;'].some(errMessage => e.message.includes(errMessage))) {
        args.result.errors.push(e.message);
        return;
      }
      retryBatchSize = retryBatchSize / 2;
      ruleExecutionLogger.warn(`New terms query for multiple fields failed due to too many clauses in query: ${e.message}. Retrying #${retryCount} with ${retryBatchSize} for composite aggregation`);
      throw e;
    }
  }, {
    retries: 2
  });
};
exports.multiTermsComposite = multiTermsComposite;