"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchRulesByQueryOrIds = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _common = require("@kbn/alerting-plugin/common");
var _find_rules = require("../../../logic/search/find_rules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fetchRulesByQueryOrIds = async ({
  query,
  ids,
  rulesClient,
  maxRules,
  gapRange
}) => {
  if (ids) {
    const fallbackErrorMessage = 'Error resolving the rule';
    try {
      const {
        rules,
        errors
      } = await rulesClient.bulkGetRules({
        ids
      });
      return {
        results: rules.map(rule => ({
          item: rule.id,
          result: rule
        })),
        errors: errors.map(({
          id,
          error
        }) => {
          let message = fallbackErrorMessage;
          if (error.statusCode === 404) {
            message = 'Rule not found';
          }
          return {
            item: id,
            error: new Error(message)
          };
        })
      };
    } catch (error) {
      // When there is an authorization error or it doesn't resolve any rule,
      // bulkGetRules will not return a partial object but throw an error instead.
      let message = error.message || fallbackErrorMessage;
      if (error.message === 'No rules found for bulk get') {
        message = 'Rule not found';
      }
      return {
        results: [],
        errors: ids.map(id => {
          return {
            item: id,
            // We do this to remove any status code set by the bulkGetRules client
            error: new Error(message)
          };
        })
      };
    }
  }
  let ruleIdsWithGaps;
  // If there is a gap range, we need to find the rules that have gaps in that range
  if (gapRange) {
    const ruleIdsWithGapsResponse = await rulesClient.getRuleIdsWithGaps({
      start: gapRange.start,
      end: gapRange.end,
      statuses: [_common.gapStatus.UNFILLED, _common.gapStatus.PARTIALLY_FILLED],
      hasUnfilledIntervals: true
    });
    ruleIdsWithGaps = ruleIdsWithGapsResponse.ruleIds;
    if (ruleIdsWithGaps.length === 0) {
      return {
        results: [],
        errors: []
      };
    }
  }
  const {
    data,
    total
  } = await (0, _find_rules.findRules)({
    rulesClient,
    perPage: maxRules,
    filter: query,
    page: undefined,
    sortField: undefined,
    sortOrder: undefined,
    fields: undefined,
    ruleIds: ruleIdsWithGaps
  });
  if (total > maxRules) {
    throw new _securitysolutionEsUtils.BadRequestError(`More than ${maxRules} rules matched the filter query. Try to narrow it down.`);
  }
  return {
    results: data.map(rule => ({
      item: rule.id,
      result: rule
    })),
    errors: []
  };
};
exports.fetchRulesByQueryOrIds = fetchRulesByQueryOrIds;