"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchActionRequests = void 0;
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/fleet-plugin/common");
var _common2 = require("@kbn/spaces-plugin/common");
var _constants = require("../constants");
var _reference_data = require("../../../lib/reference_data");
var _crowdstrike = require("../../../../../common/endpoint/service/response_actions/crowdstrike");
var _stringify = require("../../../utils/stringify");
var _ = require("../..");
var _constants2 = require("../../../../../common/endpoint/constants");
var _utils = require("../../../utils");
var _microsoft_defender = require("../../../../../common/endpoint/service/response_actions/microsoft_defender");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetches a list of Action Requests from the Endpoint action request index (not fleet)
 * @param logger
 * @param agentTypes
 * @param commands
 * @param elasticAgentIds
 * @param esClient
 * @param endDate
 * @param from
 * @param size
 * @param startDate
 * @param userIds
 * @param unExpiredOnly
 * @param types
 */
const fetchActionRequests = async ({
  endpointService,
  spaceId,
  from = 0,
  size = 10,
  agentTypes,
  commands,
  elasticAgentIds,
  endDate,
  startDate,
  userIds,
  unExpiredOnly = false,
  types
}) => {
  var _actionRequests$hits, _actionRequests$hits$, _actionRequests$hits$2, _actionRequests$hits2;
  const esClient = endpointService.getInternalEsClient();
  const logger = endpointService.createLogger('FetchActionRequests');
  const fleetServices = endpointService.getInternalFleetServices(spaceId);
  const additionalFilters = [];
  const orphanActionsSpaceId = (await endpointService.getReferenceDataClient().get(_reference_data.REF_DATA_KEYS.orphanResponseActionsSpace)).metadata.spaceId;
  if (commands !== null && commands !== void 0 && commands.length) {
    additionalFilters.push({
      terms: {
        'data.command': commands
      }
    });
  }
  if (agentTypes !== null && agentTypes !== void 0 && agentTypes.length) {
    additionalFilters.push({
      terms: {
        input_type: agentTypes
      }
    });
  }
  if (elasticAgentIds !== null && elasticAgentIds !== void 0 && elasticAgentIds.length) {
    additionalFilters.push({
      terms: {
        agents: elasticAgentIds
      }
    });
  }
  if (unExpiredOnly) {
    additionalFilters.push({
      range: {
        expiration: {
          gte: 'now'
        }
      }
    });
  }
  const must = [];

  // if space awareness is enabled, then add filter for integration policy ids
  logger.debug(() => `Space awareness is enabled - adding filter to narrow results to only response actions visible in space [${spaceId}]`);
  const matchIntegrationPolicyIds = {
    terms: {
      'agent.policy.integrationPolicyId': await fetchIntegrationPolicyIds(fleetServices)
    }
  };
  const matchOrphanActions = orphanActionsSpaceId && orphanActionsSpaceId === spaceId ? {
    term: {
      tags: _constants.ALLOWED_ACTION_REQUEST_TAGS.integrationPolicyDeleted
    }
  } : undefined;
  if (matchOrphanActions) {
    must.push({
      bool: {
        filter: {
          bool: {
            should: [matchIntegrationPolicyIds, matchOrphanActions],
            minimum_should_match: 1
          }
        }
      }
    });
  } else {
    must.push({
      bool: {
        filter: matchIntegrationPolicyIds
      }
    });
  }

  // Add the date filters
  must.push({
    bool: {
      filter: [...(0, _.getDateFilters)({
        startDate,
        endDate
      }), ...additionalFilters]
    }
  });
  if (userIds !== null && userIds !== void 0 && userIds.length) {
    const userIdsKql = userIds.map(userId => `user_id:${userId}`).join(' or ');
    const mustClause = (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(userIdsKql));
    must.push(mustClause);
  }
  const isNotASingleActionType = !types || types && types.length > 1;
  const actionsSearchQuery = {
    index: _constants2.ENDPOINT_ACTIONS_INDEX,
    size,
    from,
    query: {
      bool: {
        must,
        ...(isNotASingleActionType ? {} : getActionTypeFilter(types[0]))
      }
    },
    sort: [{
      '@timestamp': {
        order: 'desc'
      }
    }]
  };
  const actionRequests = await esClient.search(actionsSearchQuery, {
    ignore: [404]
  }).catch(_utils.catchAndWrapError);
  const total = (_actionRequests$hits = actionRequests.hits) === null || _actionRequests$hits === void 0 ? void 0 : (_actionRequests$hits$ = _actionRequests$hits.total) === null || _actionRequests$hits$ === void 0 ? void 0 : _actionRequests$hits$.value;
  logger.debug(`Searching for action requests found a total of [${total}] records using search query:\n${(0, _stringify.stringify)(actionsSearchQuery, 15)}`);
  return {
    data: ((_actionRequests$hits$2 = actionRequests === null || actionRequests === void 0 ? void 0 : (_actionRequests$hits2 = actionRequests.hits) === null || _actionRequests$hits2 === void 0 ? void 0 : _actionRequests$hits2.hits) !== null && _actionRequests$hits$2 !== void 0 ? _actionRequests$hits$2 : []).map(esHit => {
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      const action = esHit._source;

      // Ensure `agent.policy` is an array (associated with spaces)
      if (!Array.isArray(action.agent.policy)) {
        action.agent.policy = action.agent.policy ? [action.agent.policy] : [];
      }

      // Ensure `tags` is an array
      if (!Array.isArray(action.tags)) {
        action.tags = action.tags ? [action.tags] : [];
      }

      // Ensure that `originSpaceId` is populated (associated with spaces)
      if (!action.originSpaceId) {
        action.originSpaceId = _common2.DEFAULT_SPACE_ID;
      }
      return action;
    }),
    size,
    from,
    total
  };
};

/** @internal */
exports.fetchActionRequests = fetchActionRequests;
const getActionTypeFilter = actionType => {
  return actionType === 'manual' ? {
    must_not: {
      exists: {
        field: 'data.alert_id'
      }
    }
  } : actionType === 'automated' ? {
    filter: {
      exists: {
        field: 'data.alert_id'
      }
    }
  } : {};
};

/**
 * Retrieves a list of all integration policy IDs in the active space for integrations that
 * support responses actions.
 * @internal
 * @param fleetServices
 */
const fetchIntegrationPolicyIds = async fleetServices => {
  const packageNames = ['endpoint', 'sentinel_one', ...Object.keys(_crowdstrike.CROWDSTRIKE_INDEX_PATTERNS_BY_INTEGRATION), ...Object.keys(_microsoft_defender.MICROSOFT_DEFENDER_INDEX_PATTERNS_BY_INTEGRATION)];
  const kuery = `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name: (${packageNames.join(' OR ')})`;
  fleetServices.logger.debug(() => `fetchIntegrationPolicyIds(): fetching from fleet using kuery:\n${kuery}`);
  const packagePolicyIterable = await fleetServices.packagePolicy.fetchAllItemIds(fleetServices.getSoClient(), {
    kuery
  });
  const response = [];
  for await (const idList of packagePolicyIterable) {
    response.push(...idList);
  }
  fleetServices.logger.debug(() => `fetchIntegrationPolicyIds() found:\n${(0, _stringify.stringify)(response)}`);
  return response;
};