"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchActionRequestById = void 0;
var _constants = require("../constants");
var _stringify = require("../../../utils/stringify");
var _errors = require("../../../errors");
var _constants2 = require("../../../../../common/endpoint/constants");
var _utils = require("../../../utils");
var _custom_http_request_error = require("../../../../utils/custom_http_request_error");
var _reference_data = require("../../../lib/reference_data");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetches a single Action request document.
 * @param endpointService
 * @param spaceId
 * @param actionId
 * @throws
 */
const fetchActionRequestById = async (endpointService, spaceId, actionId, {
  bypassSpaceValidation = false
} = {}) => {
  var _searchResponse$hits$, _searchResponse$hits$2;
  const logger = endpointService.createLogger('fetchActionRequestById');
  const searchResponse = await endpointService.getInternalEsClient().search({
    index: _constants2.ENDPOINT_ACTIONS_INDEX,
    query: {
      bool: {
        filter: [{
          term: {
            action_id: actionId
          }
        }]
      }
    },
    size: 1
  }, {
    ignore: [404]
  }).catch(_utils.catchAndWrapError);
  const actionRequest = (_searchResponse$hits$ = searchResponse.hits.hits) === null || _searchResponse$hits$ === void 0 ? void 0 : (_searchResponse$hits$2 = _searchResponse$hits$[0]) === null || _searchResponse$hits$2 === void 0 ? void 0 : _searchResponse$hits$2._source;
  if (!actionRequest) {
    throw new _errors.NotFoundError(`Action with id '${actionId}' not found.`);
  } else if (!bypassSpaceValidation) {
    if (!actionRequest.agent.policy || actionRequest.agent.policy.length === 0) {
      const message = `Response action [${actionId}] missing 'agent.policy' information - unable to determine if response action is accessible for space [${spaceId}]`;
      logger.warn(message);
      logger.debug(`Action missing policy info:\n${(0, _stringify.stringify)(actionRequest)}`);
      throw new _custom_http_request_error.CustomHttpRequestError(message);
    } else {
      // Validate that action is visible in active space. In order for a user to be able to access
      // this response action, **at least 1** integration policy in the action request must be
      // accessible in active space
      const integrationPolicyIds = actionRequest.agent.policy.map(({
        integrationPolicyId
      }) => integrationPolicyId);
      try {
        await endpointService.getInternalFleetServices(spaceId).ensureInCurrentSpace({
          integrationPolicyIds,
          options: {
            matchAll: false
          }
        });
      } catch (err) {
        logger.debug(() => `Validation of action '${actionId}' integration policies [${integrationPolicyIds.join(', ')}] failed with: ${err.message}`);
        let throwNotFoundError = true;

        // Before throwing an error, lets check if the action has an "integration policy deleted" tag, and
        // if so, then check if its allowed to be shown in the active space
        if (actionRequest.tags && actionRequest.tags.includes(_constants.ALLOWED_ACTION_REQUEST_TAGS.integrationPolicyDeleted)) {
          logger.debug(`Checking to see if Orphan action [${actionId}] can be displayed in space [${spaceId}]`);
          const orphanActionsSpaceId = (await endpointService.getReferenceDataClient().get(_reference_data.REF_DATA_KEYS.orphanResponseActionsSpace)).metadata.spaceId;
          if (orphanActionsSpaceId && orphanActionsSpaceId === spaceId) {
            logger.debug(`Action [${actionId}] can be returned for spaceId [${spaceId}]`);
            throwNotFoundError = false;
          }
        }
        if (throwNotFoundError) {
          throw new _errors.NotFoundError(`Action [${actionId}] not found`);
        }
      }
    }
  }

  // Ensure `agent.policy` is an array
  if (!Array.isArray(actionRequest.agent.policy)) {
    actionRequest.agent.policy = actionRequest.agent.policy ? [actionRequest.agent.policy] : [];
  }

  // Ensure `tags` is an array
  if (!Array.isArray(actionRequest.tags)) {
    actionRequest.tags = actionRequest.tags ? [actionRequest.tags] : [];
  }
  return actionRequest;
};
exports.fetchActionRequestById = fetchActionRequestById;