"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MicrosoftDefenderEndpointActionsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _microsoft_defender_endpoint = require("@kbn/connector-schemas/microsoft_defender_endpoint");
var _lodash = require("lodash");
var _stream = require("stream");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _uuid = require("uuid");
var _index_name_utilities = require("../../../../../../../../common/endpoint/utils/index_name_utilities");
var _microsoft_defender = require("../../../../../../../../common/endpoint/service/response_actions/microsoft_defender");
var _base_response_actions_client = require("../../../lib/base_response_actions_client");
var _stringify = require("../../../../../../utils/stringify");
var _errors = require("../../../errors");
var _utils = require("../../../../../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable complexity */

/**
 * Validation result for MDE action details
 */

const MDE_ACTION_FETCH_RETRY_CONFIG = {
  retries: 5,
  minTimeout: 300,
  maxTimeout: 1500,
  factor: 1.5
};

// max size of script output file to retrieve
// little more than endpoint execute stderr/stdout threshold of 2kB
// to account for mde output file containing extra metadata and both stderr and stdout
const RUNSCRIPT_OUTPUT_FILE_MAX_SIZE_BYTES = 4.5 * 1024; // 4.5 KB

class MicrosoftDefenderEndpointActionsClient extends _base_response_actions_client.ResponseActionsClientImpl {
  constructor({
    connectorActions,
    ...options
  }) {
    super(options);
    (0, _defineProperty2.default)(this, "agentType", 'microsoft_defender_endpoint');
    (0, _defineProperty2.default)(this, "connectorActionsClient", void 0);
    this.connectorActionsClient = connectorActions;
    connectorActions.setup(_microsoft_defender_endpoint.CONNECTOR_ID);
  }

  /**
   * Returns a list of all indexes for Microsoft Defender data supported for response actions
   * @internal
   */
  async fetchIndexNames() {
    const cachedInfo = this.cache.get('fetchIndexNames');
    if (cachedInfo) {
      this.log.debug(`Returning cached response with list of index names:\n${(0, _stringify.stringify)(cachedInfo)}`);
      return cachedInfo;
    }
    const integrationNames = Object.keys(_microsoft_defender.MICROSOFT_DEFENDER_INDEX_PATTERNS_BY_INTEGRATION);
    const fleetServices = this.options.endpointService.getInternalFleetServices(this.options.spaceId);
    const indexNamespaces = await fleetServices.getIntegrationNamespaces(integrationNames);
    const indexNames = [];
    for (const [integrationName, namespaces] of Object.entries(indexNamespaces)) {
      if (namespaces.length > 0) {
        const indexPatterns = _microsoft_defender.MICROSOFT_DEFENDER_INDEX_PATTERNS_BY_INTEGRATION[integrationName];
        for (const indexPattern of indexPatterns) {
          indexNames.push(...namespaces.map(namespace => (0, _index_name_utilities.buildIndexNameWithNamespace)(indexPattern, namespace)));
        }
      }
    }
    this.cache.set('fetchIndexNames', indexNames);
    this.log.debug(() => `MS Defender indexes with namespace:\n${(0, _stringify.stringify)(indexNames)}`);
    return indexNames;
  }
  async fetchAgentPolicyInfo(agentIds) {
    var _msDefenderLogEsResul;
    const cacheKey = `fetchAgentPolicyInfo:${agentIds.sort().join('#')}`;
    const cacheResponse = this.cache.get(cacheKey);
    if (cacheResponse) {
      this.log.debug(() => `Cached agent policy info. found - returning it:\n${(0, _stringify.stringify)(cacheResponse)}`);
      return cacheResponse;
    }
    const esClient = this.options.esClient;
    const esSearchRequest = {
      index: await this.fetchIndexNames(),
      query: {
        bool: {
          filter: [{
            terms: {
              'cloud.instance.id': agentIds
            }
          }]
        }
      },
      collapse: {
        field: 'cloud.instance.id',
        inner_hits: {
          name: 'most_recent',
          size: 1,
          _source: ['agent', 'cloud.instance.id', 'event.created'],
          sort: [{
            'event.created': 'desc'
          }]
        }
      },
      _source: false,
      ignore_unavailable: true
    };
    if (!esSearchRequest.index || esSearchRequest.index.length === 0) {
      throw new _errors.ResponseActionsClientError(`Unable to build list of indexes while retrieving policy information for Microsoft Defender agents [${agentIds.join(', ')}]. Check to ensure at least one integration policy exists.`, 400);
    }
    this.log.debug(() => `Searching for agents with:\n${(0, _stringify.stringify)(esSearchRequest)}`);
    const msDefenderLogEsResults = await esClient.search(esSearchRequest).catch(_utils.catchAndWrapError);
    this.log.debug(() => `MS Defender Log records found:\n${(0, _stringify.stringify)(msDefenderLogEsResults, 20)}`);
    const agentIdsFound = [];
    const fleetAgentIdToMsDefenderAgentIdMap = ((_msDefenderLogEsResul = msDefenderLogEsResults.hits.hits) !== null && _msDefenderLogEsResul !== void 0 ? _msDefenderLogEsResul : []).reduce((acc, esDoc) => {
      var _esDoc$inner_hits;
      const doc = (_esDoc$inner_hits = esDoc.inner_hits) === null || _esDoc$inner_hits === void 0 ? void 0 : _esDoc$inner_hits.most_recent.hits.hits[0]._source;
      if (doc) {
        agentIdsFound.push(doc.cloud.instance.id);
        acc[doc.agent.id] = doc.cloud.instance.id;
      }
      return acc;
    }, {});
    const elasticAgentIds = Object.keys(fleetAgentIdToMsDefenderAgentIdMap);
    if (elasticAgentIds.length === 0) {
      throw new _errors.ResponseActionsClientError(`Unable to find Elastic agent IDs for Microsoft Defender agent ids: [${agentIds.join(', ')}]`, 400);
    }

    // ensure all MS agent ids were found
    for (const agentId of agentIds) {
      if (!agentIdsFound.includes(agentId)) {
        throw new _errors.ResponseActionsClientError(`Microsoft Defender agent id [${agentId}] not found`, 404);
      }
    }
    const agentPolicyInfo = await this.fetchFleetInfoForAgents(elasticAgentIds);
    for (const agentInfo of agentPolicyInfo) {
      agentInfo.agentId = fleetAgentIdToMsDefenderAgentIdMap[agentInfo.elasticAgentId];
    }
    this.cache.set(cacheKey, agentPolicyInfo);
    return agentPolicyInfo;
  }
  async handleResponseActionCreation(actionRequestOptions) {
    const actionRequestDoc = await this.writeActionRequestToEndpointIndex(actionRequestOptions);
    await this.updateCases({
      command: actionRequestOptions.command,
      caseIds: actionRequestOptions.case_ids,
      alertIds: actionRequestOptions.alert_ids,
      actionId: actionRequestDoc.EndpointActions.action_id,
      hosts: actionRequestOptions.endpoint_ids.map(agentId => {
        var _actionRequestDoc$End, _actionRequestDoc$End2;
        return {
          hostId: agentId,
          hostname: (_actionRequestDoc$End = (_actionRequestDoc$End2 = actionRequestDoc.EndpointActions.data.hosts) === null || _actionRequestDoc$End2 === void 0 ? void 0 : _actionRequestDoc$End2[agentId].name) !== null && _actionRequestDoc$End !== void 0 ? _actionRequestDoc$End : ''
        };
      }),
      comment: actionRequestOptions.comment
    });
    return {
      actionEsDoc: actionRequestDoc,
      actionDetails: await this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id)
    };
  }

  /**
   * Sends actions to Ms Defender for Endpoint directly (via Connector)
   * @internal
   */
  async sendAction(actionType, actionParams) {
    const executeOptions = {
      params: {
        subAction: actionType,
        subActionParams: actionParams
      }
    };
    this.log.debug(() => `calling connector actions 'execute()' for Microsoft Defender for Endpoint with:\n${(0, _stringify.stringify)(executeOptions)}`);
    const actionSendResponse = await this.connectorActionsClient.execute(executeOptions);
    if (actionSendResponse.status === 'error') {
      this.log.error((0, _stringify.stringify)(actionSendResponse));
      throw new _errors.ResponseActionsClientError(`Attempt to send [${actionType}] to Microsoft Defender for Endpoint failed: ${actionSendResponse.serviceMessage || actionSendResponse.message}`, 500, actionSendResponse);
    }
    this.log.debug(() => `Response:\n${(0, _stringify.stringify)(actionSendResponse)}`);
    return actionSendResponse;
  }
  async writeActionRequestToEndpointIndex(actionRequest) {
    const agentId = actionRequest.endpoint_ids[0];
    const agentDetails = await this.getAgentDetails(agentId);
    const doc = await super.writeActionRequestToEndpointIndex({
      ...actionRequest,
      hosts: {
        [agentId]: {
          name: agentDetails.computerDnsName
        }
      }
    });
    return doc;
  }

  /** Gets agent details directly from MS Defender for Endpoint */
  async getAgentDetails(agentId) {
    const cacheKey = `getAgentDetails:${agentId}`;
    const cachedEntry = this.cache.get(cacheKey);
    if (cachedEntry) {
      this.log.debug(`Found cached agent details for UUID [${agentId}]:\n${(0, _stringify.stringify)(cachedEntry)}`);
      return cachedEntry;
    }
    let msDefenderEndpointGetMachineDetailsApiResponse;
    try {
      const agentDetailsResponse = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.GET_AGENT_DETAILS, {
        id: agentId
      });
      msDefenderEndpointGetMachineDetailsApiResponse = agentDetailsResponse.data;
    } catch (err) {
      throw new _errors.ResponseActionsClientError(`Error while attempting to retrieve Microsoft Defender for Endpoint host with agent id [${agentId}]: ${err.message}`, 500, err);
    }
    if (!msDefenderEndpointGetMachineDetailsApiResponse) {
      throw new _errors.ResponseActionsClientError(`Microsoft Defender for Endpoint agent id [${agentId}] not found`, 404);
    }
    this.cache.set(cacheKey, msDefenderEndpointGetMachineDetailsApiResponse);
    return msDefenderEndpointGetMachineDetailsApiResponse;
  }

  /**
   * Validates that an MDE runscript action matches the expected script name and action ID.
   * This detects when MDE throttles/replaces our action with an existing one.
   *
   * @param actionDetails The action details returned by MDE
   * @param expectedScriptName The script name we requested
   * @param expectedActionId The Kibana action ID we included in the comment
   * @returns Validation result with isValid flag and error message if validation fails
   * @internal
   */
  checkRunscriptActionMatches(actionDetails, expectedScriptName, expectedActionId) {
    var _actionDetails$comman, _commandEntry$command, _actionDetails$reques;
    // Validate inputs
    if (!expectedScriptName || !expectedActionId) {
      return {
        isValid: false,
        error: 'Unable to validate action. Missing required parameters.'
      };
    }
    const commandEntry = (_actionDetails$comman = actionDetails.commands) === null || _actionDetails$comman === void 0 ? void 0 : _actionDetails$comman[0];
    if (!commandEntry || !((_commandEntry$command = commandEntry.command) !== null && _commandEntry$command !== void 0 && _commandEntry$command.params)) {
      return {
        isValid: false,
        error: 'Unable to verify action details. The action information is incomplete.'
      };
    }
    const scriptNameParam = commandEntry.command.params.find(p => p.key === 'ScriptName');
    const actualScriptName = scriptNameParam === null || scriptNameParam === void 0 ? void 0 : scriptNameParam.value;
    // Validate script name exists
    if (!actualScriptName) {
      return {
        isValid: false,
        error: 'Unable to verify which script is running. The action information is incomplete.'
      };
    }

    // Validate action ID is in comment
    if (!((_actionDetails$reques = actionDetails.requestorComment) !== null && _actionDetails$reques !== void 0 && _actionDetails$reques.includes(expectedActionId))) {
      return {
        isValid: false,
        error: `Cannot run script '${actualScriptName}' because an identical script is already in progress on this host (MDE action ID: ${actionDetails.id}). Please wait for the current script to complete or cancel it before trying again.`
      };
    }

    // Validate script name matches
    if (actualScriptName !== expectedScriptName) {
      return {
        isValid: false,
        error: `Cannot run script '${expectedScriptName}' because another script ('${actualScriptName}') is already in progress on this host (MDE action ID: ${actionDetails.id}). Please wait for the current script to complete or cancel it before trying again.`
      };
    }

    // All validations passed - this is our action
    return {
      isValid: true
    };
  }

  /**
   * Fetches and validates the details of a specific runscript action from Microsoft Defender for Endpoint.
   * This method ensures that the action returned by MDE matches our expected script name and action ID,
   * detecting when MDE throttles/replaces our action with an existing one.
   *
   * @param machineActionId - The Microsoft Defender machine action ID returned from sendAction
   * @param expectedScriptName - The script name we requested to run
   * @param expectedActionId - The Kibana action ID we included in the comment
   * @returns Validation result with isValid flag and error message if validation fails
   * @internal
   */
  async fetchAndValidateRunscriptActionDetails(machineActionId, expectedScriptName, expectedActionId) {
    this.log.debug(`Fetching action details from MDE API for machineActionId [${machineActionId}]`);
    try {
      // Retry fetching action details to handle MDE API indexing lag.
      // When MDE accepts a new action, there's a delay before it appears in their GET actions API.
      // We retry request with linear increase in time delay (300ms → 450ms → 675ms → 1012ms → 1518ms) up to 5 times
      // to give MDE's internal indexing sufficient time to make the action available.
      const actionDetails = await (0, _pRetry.default)(async () => {
        var _response$data, _response$data$value;
        this.log.debug(`Attempting to fetch MDE action [${machineActionId}]`);
        const params = {
          id: [machineActionId],
          pageSize: 1
        };
        const response = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.GET_ACTIONS, params);
        const action = (_response$data = response.data) === null || _response$data === void 0 ? void 0 : (_response$data$value = _response$data.value) === null || _response$data$value === void 0 ? void 0 : _response$data$value[0];
        if (!action) {
          throw new Error(`Action not yet available in MDE API for machineActionId [${machineActionId}]`);
        }
        return action;
      }, {
        ...MDE_ACTION_FETCH_RETRY_CONFIG,
        onFailedAttempt: ({
          attemptNumber,
          retriesLeft,
          message
        }) => {
          this.log.debug(`Attempt ${attemptNumber} to fetch MDE action [${machineActionId}] failed. ${retriesLeft} retries left. [ERROR: ${message}]`);
        }
      });
      this.log.debug(`Successfully fetched action details for machineActionId [${machineActionId}]: status=${actionDetails.status}, type=${actionDetails.type}`);

      // Validate if the response action matches the MDE action
      return this.checkRunscriptActionMatches(actionDetails, expectedScriptName, expectedActionId);
    } catch (error) {
      this.log.error(`Failed to fetch action details from MDE API for machineActionId [${machineActionId}]: ${error.message}`);
      return {
        isValid: false,
        error: `Action details not found in Microsoft Defender for machineActionId [${machineActionId}]. The action may not have been created successfully.`
      };
    }
  }
  async validateRequest(
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  payload) {
    // TODO: support multiple agents
    if (payload.endpoint_ids.length > 1) {
      return {
        isValid: false,
        error: new _errors.ResponseActionsClientError(`[body.endpoint_ids]: Multiple agents IDs not currently supported for Microsoft Defender for Endpoint`, 400)
      };
    }

    // For cancel actions, perform comprehensive validation
    if (payload.command === 'cancel') {
      var _payload$parameters;
      const {
        microsoftDefenderEndpointCancelEnabled
      } = this.options.endpointService.experimentalFeatures;
      if (!microsoftDefenderEndpointCancelEnabled) {
        throw new _errors.ResponseActionsClientError('Cancel operation is not enabled for Microsoft Defender for Endpoint', 400);
      }
      const actionId = (_payload$parameters = payload.parameters) === null || _payload$parameters === void 0 ? void 0 : _payload$parameters.id;
      if (!actionId) {
        return {
          isValid: false,
          error: new _errors.ResponseActionsClientError('id is required in parameters for cancel action', 400)
        };
      }
      try {
        var _payload$endpoint_ids;
        // Fetch the original action to validate cancel request
        const originalAction = await this.fetchActionDetails(actionId);

        // Check if action is already completed
        if (originalAction.isCompleted) {
          const statusMessage = originalAction.wasSuccessful ? 'completed successfully' : 'failed';
          return {
            isValid: false,
            error: new _errors.ResponseActionsClientError(`Cannot cancel action [${actionId}] because it has already ${statusMessage}.`, 400)
          };
        }

        // Validate endpoint ID association if provided
        const requestEndpointId = (_payload$endpoint_ids = payload.endpoint_ids) === null || _payload$endpoint_ids === void 0 ? void 0 : _payload$endpoint_ids[0];
        if (requestEndpointId && originalAction.agents) {
          const originalActionAgentIds = Array.isArray(originalAction.agents) ? originalAction.agents : [originalAction.agents];
          if (!originalActionAgentIds.includes(requestEndpointId)) {
            return {
              isValid: false,
              error: new _errors.ResponseActionsClientError(`Endpoint '${requestEndpointId}' is not associated with action '${actionId}'`, 400)
            };
          }
        }

        // Validate command information exists
        if (!originalAction.command) {
          return {
            isValid: false,
            error: new _errors.ResponseActionsClientError(`Unable to determine command type for action '${actionId}'`, 500)
          };
        }

        // Check if we're trying to cancel a cancel action (business rule validation)
        if (originalAction.command === 'cancel') {
          return {
            isValid: false,
            error: new _errors.ResponseActionsClientError(`Cannot cancel a cancel action.`, 400)
          };
        }
      } catch (error) {
        // If we can't fetch the action details (e.g., action not found),
        // return a validation error
        if (error instanceof Error && error.message.includes('not found')) {
          return {
            isValid: false,
            error: new _errors.ResponseActionsClientError(`Action with id '${actionId}' not found.`, 404)
          };
        }
        // For other errors, let them bubble up
        throw error;
      }
    }
    return super.validateRequest(payload);
  }
  async isolate(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      parameters: undefined,
      command: 'isolate'
    };
    if (!reqIndexOptions.error) {
      var _error2;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          var _msActionResponse$dat;
          const msActionResponse = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.ISOLATE_HOST, {
            id: actionRequest.endpoint_ids[0],
            comment: this.buildExternalComment(reqIndexOptions)
          });
          if (msActionResponse !== null && msActionResponse !== void 0 && (_msActionResponse$dat = msActionResponse.data) !== null && _msActionResponse$dat !== void 0 && _msActionResponse$dat.id) {
            reqIndexOptions.meta = {
              machineActionId: msActionResponse.data.id
            };
          } else {
            throw new _errors.ResponseActionsClientError(`Isolate request was sent to Microsoft Defender, but Machine Action Id was not provided!`);
          }
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error2 = error) === null || _error2 === void 0 ? void 0 : _error2.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async release(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      parameters: undefined,
      command: 'unisolate'
    };
    if (!reqIndexOptions.error) {
      var _error3;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          var _msActionResponse$dat2;
          const msActionResponse = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.RELEASE_HOST, {
            id: actionRequest.endpoint_ids[0],
            comment: this.buildExternalComment(reqIndexOptions)
          });
          if (msActionResponse !== null && msActionResponse !== void 0 && (_msActionResponse$dat2 = msActionResponse.data) !== null && _msActionResponse$dat2 !== void 0 && _msActionResponse$dat2.id) {
            reqIndexOptions.meta = {
              machineActionId: msActionResponse.data.id
            };
          } else {
            throw new _errors.ResponseActionsClientError(`Un-Isolate request was sent to Microsoft Defender, but Machine Action Id was not provided!`);
          }
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error3 = error) === null || _error3 === void 0 ? void 0 : _error3.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async runscript(actionRequest, options) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      parameters: actionRequest.parameters,
      command: 'runscript'
    };
    const {
      scriptName,
      args
    } = reqIndexOptions.parameters;
    if (!reqIndexOptions.error) {
      var _reqValidationError;
      let reqValidationError = (await this.validateRequest(reqIndexOptions)).error;
      if (!reqValidationError) {
        try {
          var _msActionResponse$dat3;
          const msActionResponse = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.RUN_SCRIPT, {
            id: reqIndexOptions.endpoint_ids[0],
            comment: this.buildExternalComment(reqIndexOptions),
            parameters: {
              scriptName,
              args
            }
          });
          const machineActionId = msActionResponse === null || msActionResponse === void 0 ? void 0 : (_msActionResponse$dat3 = msActionResponse.data) === null || _msActionResponse$dat3 === void 0 ? void 0 : _msActionResponse$dat3.id;
          if (!machineActionId) {
            throw new _errors.ResponseActionsClientError(`Run Script request was sent to Microsoft Defender, but Machine Action Id was not provided!`);
          }

          // Ensure actionId is set for validation. While buildExternalComment() should have already
          // set it via side effect, TypeScript doesn't track this, and defensive programming dictates
          // we guarantee it exists.
          if (!reqIndexOptions.actionId) {
            reqIndexOptions.actionId = (0, _uuid.v4)();
          }
          const mdeActionValidation = await this.fetchAndValidateRunscriptActionDetails(machineActionId, scriptName, reqIndexOptions.actionId);
          if (!mdeActionValidation.isValid) {
            var _mdeActionValidation$;
            throw new _errors.ResponseActionsClientError((_mdeActionValidation$ = mdeActionValidation.error) !== null && _mdeActionValidation$ !== void 0 ? _mdeActionValidation$ : 'A runscript action is already pending in MS Defender.', 409, {
              machineActionId,
              requestedScript: scriptName
            });
          }
          reqIndexOptions.meta = {
            machineActionId
          };
        } catch (err) {
          reqValidationError = err;
        }
      }
      reqIndexOptions.error = (_reqValidationError = reqValidationError) === null || _reqValidationError === void 0 ? void 0 : _reqValidationError.message;
      if (!this.options.isAutomated && reqValidationError) {
        throw reqValidationError;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async cancel(actionRequest, options = {}) {
    var _actionRequest$parame;
    const actionId = (_actionRequest$parame = actionRequest.parameters) === null || _actionRequest$parame === void 0 ? void 0 : _actionRequest$parame.id;
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      command: 'cancel',
      parameters: {
        id: actionId
      }
    };
    if (!reqIndexOptions.error) {
      var _error4;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          var _actionRequestWithExt, _msActionResponse$dat4;
          // Get the external action ID from the internal response action ID
          const actionRequestWithExternalId = await this.fetchActionRequestEsDoc(actionId);
          const externalActionId = actionRequestWithExternalId === null || actionRequestWithExternalId === void 0 ? void 0 : (_actionRequestWithExt = actionRequestWithExternalId.meta) === null || _actionRequestWithExt === void 0 ? void 0 : _actionRequestWithExt.machineActionId;
          if (!externalActionId) {
            throw new _errors.ResponseActionsClientError(`Unable to resolve Microsoft Defender machine action ID for action [${actionId}]`, 500);
          }
          const msActionResponse = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.CANCEL_ACTION, {
            actionId: externalActionId,
            comment: this.buildExternalComment(reqIndexOptions)
          });
          if (msActionResponse !== null && msActionResponse !== void 0 && (_msActionResponse$dat4 = msActionResponse.data) !== null && _msActionResponse$dat4 !== void 0 && _msActionResponse$dat4.id) {
            reqIndexOptions.meta = {
              machineActionId: msActionResponse.data.id
            };
          } else {
            throw new _errors.ResponseActionsClientError(`Cancel request was sent to Microsoft Defender, but Machine Action Id was not provided!`);
          }
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error4 = error) === null || _error4 === void 0 ? void 0 : _error4.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async processPendingActions({
    abortSignal,
    addToQueue
  }) {
    if (abortSignal.aborted) {
      return;
    }
    const addResponsesToQueueIfAny = responseList => {
      if (responseList.length > 0) {
        addToQueue(...responseList);
        this.sendActionResponseTelemetry(responseList);
      }
    };
    for await (const pendingActions of this.fetchAllPendingActions()) {
      if (abortSignal.aborted) {
        return;
      }
      const pendingActionsByType = (0, _lodash.groupBy)(pendingActions, 'action.EndpointActions.data.command');
      for (const [actionType, typePendingActions] of Object.entries(pendingActionsByType)) {
        if (abortSignal.aborted) {
          return;
        }
        switch (actionType) {
          case 'isolate':
          case 'unisolate':
          case 'cancel':
          case 'runscript':
            addResponsesToQueueIfAny(await this.checkPendingActions(typePendingActions));
            break;
        }
      }
    }
  }
  async checkPendingActions(actionRequests) {
    const completedResponses = [];
    const warnings = [];
    const actionsByMachineId = {};
    const machineActionIds = [];
    const msApiOptions = {
      id: machineActionIds,
      pageSize: 1000
    };
    for (const {
      action
    } of actionRequests) {
      var _action$meta;
      const command = action.EndpointActions.data.command;
      const machineActionId = (_action$meta = action.meta) === null || _action$meta === void 0 ? void 0 : _action$meta.machineActionId;
      if (!machineActionId) {
        warnings.push(`${command} response action ID [${action.EndpointActions.action_id}] is missing Microsoft Defender for Endpoint machine action id, thus unable to check on it's status. Forcing it to complete as failure.`);
        completedResponses.push(this.buildActionResponseEsDoc({
          actionId: action.EndpointActions.action_id,
          agentId: Array.isArray(action.agent.id) ? action.agent.id[0] : action.agent.id,
          data: {
            command
          },
          error: {
            message: `Unable to verify if action completed. Microsoft Defender machine action id ('meta.machineActionId') missing from the action request document!`
          }
        }));
      } else {
        if (!actionsByMachineId[machineActionId]) {
          actionsByMachineId[machineActionId] = [];
        }
        actionsByMachineId[machineActionId].push(action);
        machineActionIds.push(machineActionId);
      }
    }
    const {
      data: machineActions
    } = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.GET_ACTIONS, msApiOptions);
    if (machineActions !== null && machineActions !== void 0 && machineActions.value) {
      for (const machineAction of machineActions.value) {
        var _machineAction$comman, _machineAction$comman2, _machineAction$comman3;
        const machineActionId = machineAction.id;
        const mdeCommand = machineAction.type;
        const {
          isPending,
          isError,
          message: machineStateMessage
        } = this.calculateMachineActionState(machineAction);
        const message = ((_machineAction$comman = machineAction.commands) === null || _machineAction$comman === void 0 ? void 0 : (_machineAction$comman2 = _machineAction$comman[0]) === null || _machineAction$comman2 === void 0 ? void 0 : (_machineAction$comman3 = _machineAction$comman2.errors) === null || _machineAction$comman3 === void 0 ? void 0 : _machineAction$comman3.join('\n')) || machineStateMessage;

        // completed actions -> failed | succeeded - build response docs for all associated action requests
        if (!isPending) {
          var _actionsByMachineId$m;
          const completeActionRequests = (_actionsByMachineId$m = actionsByMachineId[machineActionId]) !== null && _actionsByMachineId$m !== void 0 ? _actionsByMachineId$m : [];
          for (const actionRequest of completeActionRequests) {
            const actionId = actionRequest.EndpointActions.action_id;
            const command = actionRequest.EndpointActions.data.command;
            const isCancelledAction = machineAction.status === 'Cancelled' && command === 'cancel';
            // Special handling for cancelled actions:
            // Cancel actions that successfully cancel something should show as success
            // Actions that were cancelled by another action should show as failed
            const shouldUpdateErrorMessage = !isCancelledAction && isError;
            const isRunscriptAction = mdeCommand === 'LiveResponse' && command === 'runscript';
            let output;
            let meta;
            if (isRunscriptAction) {
              const outputFile = await this.fetchMachineLiveResponseFile(machineActionId);
              output = {
                type: 'json',
                content: {
                  stdout: (outputFile === null || outputFile === void 0 ? void 0 : outputFile.stdout) || '',
                  stderr: (outputFile === null || outputFile === void 0 ? void 0 : outputFile.stderr) || '',
                  code: (outputFile === null || outputFile === void 0 ? void 0 : outputFile.code) || '0'
                }
              };
              meta = {
                machineActionId,
                filename: `runscript_output_${machineActionId}_0.json`,
                createdAt: new Date().toISOString()
              };
            }
            completedResponses.push(this.buildActionResponseEsDoc({
              actionId,
              agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
              data: {
                command: actionRequest.EndpointActions.data.command,
                output
              },
              error: shouldUpdateErrorMessage ? {
                message
              } : undefined,
              meta
            }));
          }
        }
      }
    }
    this.log.debug(() => `${completedResponses.length} action responses generated:\n${(0, _stringify.stringify)(completedResponses)}`);
    if (warnings.length > 0) {
      this.log.warn(warnings.join('\n'));
    }
    return completedResponses;
  }
  calculateMachineActionState(machineAction) {
    let isPending = true;
    let isError = false;
    let message = '';
    switch (machineAction.status) {
      case 'Failed':
      case 'TimeOut':
        isPending = false;
        isError = true;
        message = `Response action ${machineAction.status} (Microsoft Defender for Endpoint machine action ID: ${machineAction.id})`;
        break;
      case 'Cancelled':
        isPending = false;
        isError = true;
        message = `Response action was canceled by [${machineAction.cancellationRequestor}] (Microsoft Defender for Endpoint machine action ID: ${machineAction.id})${machineAction.cancellationComment ? `: ${machineAction.cancellationComment}` : ''}`;
        break;
      case 'Succeeded':
        isPending = false;
        isError = false;
        break;
      default:
        // covers 'Pending' | 'InProgress'
        isPending = true;
        isError = false;
    }
    return {
      isPending,
      isError,
      message
    };
  }
  async getCustomScripts() {
    try {
      var _customScriptsRespons;
      const customScriptsResponse = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.GET_LIBRARY_FILES, {});
      const scripts = ((_customScriptsRespons = customScriptsResponse.data) === null || _customScriptsRespons === void 0 ? void 0 : _customScriptsRespons.value) || [];

      // Transform MS Defender scripts to ResponseActionScriptsApiResponse format
      const data = scripts.map(script => ({
        // due to External EDR's schema nature - we expect a maybe() everywhere - empty strings are needed
        id: script.fileName || '',
        name: script.fileName || '',
        description: script.description || ''
      }));
      return {
        data
      };
    } catch (err) {
      const error = new _errors.ResponseActionsClientError(`Failed to fetch Microsoft Defender for Endpoint scripts, failed with: ${err.message}`, 500, err);
      this.log.error(error);
      throw error;
    }
  }
  async getFileInfo(actionId, agentId) {
    await this.ensureValidActionId(actionId);
    const {
      EndpointActions: {
        data: {
          command
        }
      }
    } = await this.fetchActionRequestEsDoc(actionId);
    const {
      microsoftDefenderEndpointRunScriptEnabled
    } = this.options.endpointService.experimentalFeatures;
    if (command === 'runscript' && !microsoftDefenderEndpointRunScriptEnabled) {
      throw new _errors.ResponseActionsClientError(`File downloads are not supported for ${this.agentType} agent type. Feature disabled.`, 400);
    }
    const fileInfo = {
      actionId,
      agentId,
      id: agentId,
      agentType: this.agentType,
      status: 'AWAITING_UPLOAD',
      created: '',
      name: '',
      size: 0,
      mimeType: ''
    };
    try {
      switch (command) {
        case 'runscript':
          {
            var _agentResponse$meta$c, _agentResponse$meta, _agentResponse$meta$f, _agentResponse$meta2;
            const agentResponse = await this.fetchEsResponseDocForAgentId(actionId, agentId);
            fileInfo.status = 'READY';
            fileInfo.created = (_agentResponse$meta$c = (_agentResponse$meta = agentResponse.meta) === null || _agentResponse$meta === void 0 ? void 0 : _agentResponse$meta.createdAt) !== null && _agentResponse$meta$c !== void 0 ? _agentResponse$meta$c : '';
            fileInfo.name = (_agentResponse$meta$f = (_agentResponse$meta2 = agentResponse.meta) === null || _agentResponse$meta2 === void 0 ? void 0 : _agentResponse$meta2.filename) !== null && _agentResponse$meta$f !== void 0 ? _agentResponse$meta$f : '';
            fileInfo.mimeType = 'application/octet-stream';
          }
          break;
        default:
          throw new _errors.ResponseActionsClientError(`${command} does not support file downloads`, 400);
      }
    } catch (e) {
      // Ignore "no response doc" error for the agent and just return the file info with the status of 'AWAITING_UPLOAD'
      if (!(e instanceof _errors.ResponseActionAgentResponseEsDocNotFound)) {
        throw e;
      }
    }
    return fileInfo;
  }
  async getFileDownload(actionId, agentId) {
    await this.ensureValidActionId(actionId);
    const {
      EndpointActions: {
        data: {
          command
        }
      }
    } = await this.fetchActionRequestEsDoc(actionId);
    const {
      microsoftDefenderEndpointRunScriptEnabled
    } = this.options.endpointService.experimentalFeatures;
    if (command === 'runscript' && !microsoftDefenderEndpointRunScriptEnabled) {
      throw new _errors.ResponseActionsClientError(`File downloads are not supported for ${this.agentType} agent type. Feature disabled.`, 400);
    }
    let downloadStream;
    let fileName = 'download.json';
    try {
      switch (command) {
        case 'runscript':
          {
            var _runscriptAgentRespon, _runscriptAgentRespon2;
            const runscriptAgentResponse = await this.fetchEsResponseDocForAgentId(actionId, agentId);
            if (!((_runscriptAgentRespon = runscriptAgentResponse.meta) !== null && _runscriptAgentRespon !== void 0 && _runscriptAgentRespon.machineActionId)) {
              throw new _errors.ResponseActionsClientError(`Unable to retrieve file from Microsoft Defender for Endpoint. Response ES document is missing [meta.machineActionId]`);
            }
            const {
              data
            } = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.GET_ACTION_RESULTS, {
              id: (_runscriptAgentRespon2 = runscriptAgentResponse.meta) === null || _runscriptAgentRespon2 === void 0 ? void 0 : _runscriptAgentRespon2.machineActionId
            });
            if (data) {
              downloadStream = data;
              fileName = runscriptAgentResponse.meta.filename;
            }
          }
          break;
      }
      if (!downloadStream) {
        throw new _errors.ResponseActionsClientError(`Unable to establish a file download Readable stream with Microsoft Defender for Endpoint for response action [${command}] [${actionId}]`);
      }
    } catch (e) {
      this.log.debug(() => `Attempt to get file download stream from Microsoft Defender for Endpoint for response action failed with:\n${(0, _stringify.stringify)(e)}`);
      throw e;
    }
    return {
      stream: downloadStream,
      mimeType: undefined,
      fileName
    };
  }
  async fetchEsResponseDocForAgentId(actionId, agentId) {
    const agentResponse = (await this.fetchActionResponseEsDocs(actionId, [agentId]))[agentId];
    if (!agentResponse) {
      throw new _errors.ResponseActionAgentResponseEsDocNotFound(`Action ID [${actionId}] for agent ID [${agentId}] is still pending`, 404);
    }
    return agentResponse;
  }
  async fetchMachineLiveResponseFile(machineActionId) {
    try {
      const {
        data: readableStream
      } = await this.sendAction(_microsoft_defender_endpoint.SUB_ACTION.GET_ACTION_RESULTS, {
        id: machineActionId
      });
      if (!readableStream) {
        this.log.debug(`No download stream available for machine action ${machineActionId}`);
        return null;
      }
      if (readableStream.readableLength > RUNSCRIPT_OUTPUT_FILE_MAX_SIZE_BYTES) {
        this.log.debug(`Download stream for machine action ${machineActionId} exceeds max size of ${RUNSCRIPT_OUTPUT_FILE_MAX_SIZE_BYTES} bytes`);
        readableStream.destroy();
        return {
          stdout: '',
          stderr: 'The output file is too large to download and display.',
          code: '413'
        };
      }
      return this.processStreamOutput(readableStream);
    } catch ({
      message,
      statusCode
    }) {
      this.log.error(`Failed to fetch runscript output for machine action ${machineActionId}: ${message}`);
      return {
        stdout: '',
        stderr: message,
        code: statusCode
      };
    }
  }
  async processStreamOutput(stream) {
    try {
      let jsonStr = '';
      const writable = new _stream.Writable({
        write(chunk, _, callback) {
          jsonStr += chunk.toString('utf8');
          callback();
        }
      });
      const output = await new Promise((resolve, reject) => {
        writable.on('finish', () => {
          try {
            resolve(JSON.parse(jsonStr));
          } catch (parseError) {
            this.log.error(`Failed to parse runscript output file JSON: ${parseError.message}`);
            reject(parseError);
          }
        }).on('error', reject);
        stream.pipe(writable);
      });
      return {
        stdout: output.script_output,
        stderr: output.script_errors,
        code: output.exit_code
      };
    } catch (_error) {
      const error = `Failed to process runscript output file: ${_error.message}`;
      this.log.error(error, {
        error: _error
      });
      return {
        stdout: '',
        stderr: error,
        code: '1'
      };
    }
  }
}
exports.MicrosoftDefenderEndpointActionsClient = MicrosoftDefenderEndpointActionsClient;