"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postProtectionUpdatesNoteHandler = exports.getProtectionUpdatesNoteHandler = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _saved_object_mappings = require("../../lib/protection_updates_note/saved_object_mappings");
var _error_handler = require("../error_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getProtectionNote = async (SOClient, packagePolicyId) => {
  return SOClient.find({
    type: _saved_object_mappings.protectionUpdatesNoteSavedObjectType,
    hasReference: {
      type: _common.PACKAGE_POLICY_SAVED_OBJECT_TYPE,
      id: packagePolicyId
    }
  });
};
const updateProtectionNote = async (SOClient, noteId, note, references) => {
  return SOClient.update(_saved_object_mappings.protectionUpdatesNoteSavedObjectType, noteId, {
    note
  }, {
    references,
    refresh: 'wait_for'
  });
};
const createProtectionNote = async (SOClient, note, references) => {
  return SOClient.create(_saved_object_mappings.protectionUpdatesNoteSavedObjectType, {
    note
  }, {
    references,
    refresh: 'wait_for'
  });
};
async function getSavedObjectClient(context, endpointContext, packagePolicyId) {
  const spaceId = (await context.securitySolution).getSpaceId();
  const scopedFleetService = endpointContext.service.getInternalFleetServices(spaceId);
  await scopedFleetService.ensureInCurrentSpace({
    integrationPolicyIds: [packagePolicyId]
  });
  const unscopedFleetService = endpointContext.service.getInternalFleetServices(undefined, true);
  return unscopedFleetService.getSoClient();
}
const postProtectionUpdatesNoteHandler = function (endpointContext) {
  return async (context, request, response) => {
    const {
      package_policy_id: packagePolicyId
    } = request.params;
    let SOClient;
    let soClientResponse;
    const logger = endpointContext.logFactory.get('protectionUpdatesNote');
    try {
      SOClient = await getSavedObjectClient(context, endpointContext, packagePolicyId);
      soClientResponse = await getProtectionNote(SOClient, packagePolicyId);
    } catch (err) {
      return (0, _error_handler.errorHandler)(logger, response, err);
    }
    const {
      note
    } = request.body;
    if (soClientResponse.saved_objects[0]) {
      const {
        references
      } = soClientResponse.saved_objects[0];
      let updatedNoteSO;
      try {
        updatedNoteSO = await updateProtectionNote(SOClient, soClientResponse.saved_objects[0].id, note, references);
      } catch (err) {
        return (0, _error_handler.errorHandler)(logger, response, err);
      }
      const {
        attributes
      } = updatedNoteSO;
      return response.ok({
        body: attributes
      });
    }
    const references = [{
      id: packagePolicyId,
      name: 'package_policy',
      type: _common.PACKAGE_POLICY_SAVED_OBJECT_TYPE
    }];
    let noteSO;
    try {
      noteSO = await createProtectionNote(SOClient, note, references);
    } catch (err) {
      return (0, _error_handler.errorHandler)(logger, response, err);
    }
    const {
      attributes
    } = noteSO;
    return response.ok({
      body: attributes
    });
  };
};
exports.postProtectionUpdatesNoteHandler = postProtectionUpdatesNoteHandler;
const getProtectionUpdatesNoteHandler = function (endpointContext) {
  return async (context, request, response) => {
    const {
      package_policy_id: packagePolicyId
    } = request.params;
    let SOClient;
    let soClientResponse;
    try {
      SOClient = await getSavedObjectClient(context, endpointContext, packagePolicyId);
      soClientResponse = await getProtectionNote(SOClient, packagePolicyId);
    } catch (err) {
      const logger = endpointContext.logFactory.get('protectionUpdatesNote');
      return (0, _error_handler.errorHandler)(logger, response, err);
    }
    if (!soClientResponse.saved_objects[0] || !soClientResponse.saved_objects[0].attributes) {
      return response.notFound({
        body: {
          message: 'No note found for this policy'
        }
      });
    }
    const {
      attributes
    } = soClientResponse.saved_objects[0];
    return response.ok({
      body: attributes
    });
  };
};
exports.getProtectionUpdatesNoteHandler = getProtectionUpdatesNoteHandler;