"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReferenceDataClient = void 0;
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _errors = require("../../../../common/endpoint/errors");
var _constants = require("./constants");
var _stringify = require("../../utils/stringify");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Data client to interact with Security Solution Reference Data (saved in saved objects).
 * Reference Data is a concept of maintaining internal known items (they have unique "key"s).
 */
class ReferenceDataClient {
  constructor(soClient, logger) {
    this.soClient = soClient;
    this.logger = logger;
  }
  async create(refDataKey, data) {
    const {
      soClient,
      logger
    } = this;
    logger.debug(`creating reference data [${refDataKey}]`);
    return soClient.create(_constants.REFERENCE_DATA_SAVED_OBJECT_TYPE, data, {
      id: refDataKey,
      refresh: 'wait_for'
    }).then(response => {
      logger.debug(`Reference data item [${refDataKey}] created successfully`);
      return response.attributes;
    }).catch(error => {
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isConflictError(error)) {
        logger.debug(`Looks like reference data [${refDataKey}] already exists - return it`);
        return this.get(refDataKey);
      }
      return (0, _utils.catchAndWrapError)(error);
    });
  }
  async get(refDataKey) {
    const {
      soClient,
      logger
    } = this;
    return soClient.get(_constants.REFERENCE_DATA_SAVED_OBJECT_TYPE, refDataKey).then(response => {
      logger.debug(`Retrieved [${refDataKey}]\n${(0, _stringify.stringify)(response)}`);
      return response.attributes;
    }).catch(async err => {
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        if (_constants.REF_DATA_KEY_INITIAL_VALUE[refDataKey]) {
          return this.create(refDataKey, _constants.REF_DATA_KEY_INITIAL_VALUE[refDataKey]());
        }
        throw new _errors.EndpointError(`Definition for reference data key [${refDataKey}] not defined. Unable to create it.`);
      }
      throw (0, _utils.wrapErrorIfNeeded)(err, `Failed to retrieve reference data item [${refDataKey}]`);
    });
  }
  async update(refDataKey, data) {
    const {
      soClient,
      logger
    } = this;
    logger.debug(`updating reference data [${refDataKey}]`);
    if (data.id !== refDataKey) {
      throw new _errors.EndpointError(`Updated data 'id' value [${data.id}] differs from the reference data key provided [${refDataKey}]`);
    }
    await soClient.update(_constants.REFERENCE_DATA_SAVED_OBJECT_TYPE, refDataKey, data, {
      refresh: 'wait_for'
    }).catch(_utils.catchAndWrapError);
    return data;
  }
  async delete(refDataKey) {
    const {
      soClient,
      logger
    } = this;
    logger.debug(`Deleting reference data [${refDataKey}]`);
    await soClient.delete(_constants.REFERENCE_DATA_SAVED_OBJECT_TYPE, refDataKey).catch(error => {
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        return;
      }
      throw (0, _utils.wrapErrorIfNeeded)(error);
    });
  }
}
exports.ReferenceDataClient = ReferenceDataClient;