"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRiskScorePalette = exports.getFillColor = exports.RISK_SCORE_STEPS = void 0;
var _fp = require("lodash/fp");
var _constants = require("../../../../../../../../common/detection_engine/constants");
var _risk_color_palette = require("../../../../../../../common/utils/risk_color_palette");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The detection engine creates risk scores in the range 1 - 100.
 * These steps also include a score of "zero", to enable lookups
 * via an array index.
 */
const RISK_SCORE_STEPS = exports.RISK_SCORE_STEPS = 101;

/**
 * Returns a color palette that maps a risk score to the risk score colors
 * defined by the Security Solution.
 *
 * The pallet defines values for a risk score between 0 and 100 (inclusive),
 * but in practice, the detection engine only generates scores between 1-100.
 *
 * This pallet has the same type as `EuiPalette`, which is not exported by
 * EUI at the time of this writing.
 */
const getRiskScorePalette = (steps, euiTheme) => {
  const palette = (0, _risk_color_palette.getRiskSeverityColors)(euiTheme);
  return Array(steps).fill(0).map((_, i) => {
    if (i >= _constants.RISK_SCORE_CRITICAL) {
      return palette.critical;
    } else if (i >= _constants.RISK_SCORE_HIGH) {
      return palette.high;
    } else if (i >= _constants.RISK_SCORE_MEDIUM) {
      return palette.medium;
    } else {
      return palette.low;
    }
  });
};

/** Returns a fill color based on the index of the risk score in the color palette */
exports.getRiskScorePalette = getRiskScorePalette;
const getFillColor = ({
  riskScore,
  colorPalette
}) => {
  const MIN_RISK_SCORE = 0;
  const MAX_RISK_SCORE = Math.min(100, colorPalette.length);
  const clampedScore = (0, _fp.clamp)(MIN_RISK_SCORE, MAX_RISK_SCORE, riskScore);
  return colorPalette[clampedScore];
};
exports.getFillColor = getFillColor;