"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineCommonRoutes = defineCommonRoutes;
var _configSchema = require("@kbn/config-schema");
var _std = require("@kbn/std");
var _authentication = require("../../authentication");
var _errors = require("../../errors");
var _instrumentation = require("../../otel/instrumentation");
var _licensed_route_handler = require("../licensed_route_handler");
var _tags = require("../tags");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Defines routes that are common to various authentication mechanisms.
 */
function defineCommonRoutes({
  router,
  getAuthenticationService,
  basePath,
  license,
  logger,
  buildFlavor
}) {
  router.get({
    path: '/api/security/logout',
    security: {
      authz: {
        enabled: false,
        reason: 'This route must remain accessible to 3rd-party IdPs'
      },
      authc: {
        enabled: false,
        reason: 'This route must remain accessible to 3rd-party IdPs'
      }
    },
    // Allow unknown query parameters as this endpoint can be hit by the 3rd-party with any
    // set of query string parameters (e.g. SAML/OIDC logout request/response parameters).
    validate: {
      query: _configSchema.schema.object({}, {
        unknowns: 'allow'
      })
    },
    options: {
      access: 'public',
      excludeFromOAS: true,
      tags: [_tags.ROUTE_TAG_CAN_REDIRECT, _tags.ROUTE_TAG_AUTH_FLOW]
    }
  }, async (context, request, response) => {
    const serverBasePath = basePath.serverBasePath;
    if (!(0, _authentication.canRedirectRequest)(request)) {
      return response.badRequest({
        body: 'Client should be able to process redirect response.'
      });
    }
    try {
      const deauthenticationResult = await getAuthenticationService().logout(request);
      if (deauthenticationResult.failed()) {
        _instrumentation.securityTelemetry.recordLogoutAttempt({
          outcome: 'failure'
        });
        return response.customError((0, _errors.wrapIntoCustomErrorResponse)(deauthenticationResult.error));
      }
      _instrumentation.securityTelemetry.recordLogoutAttempt({
        outcome: 'success'
      });
      return response.redirected({
        headers: {
          location: deauthenticationResult.redirectURL || `${serverBasePath}/`
        }
      });
    } catch (error) {
      _instrumentation.securityTelemetry.recordLogoutAttempt({
        outcome: 'failure'
      });
      return response.customError((0, _errors.wrapIntoCustomErrorResponse)(error));
    }
  });
  router.get({
    path: '/internal/security/me',
    security: {
      authz: {
        enabled: false,
        reason: `This route delegates authorization to Core's security service; there must be an authenticated user for this route to return information`
      }
    },
    validate: false,
    options: {
      access: 'internal'
    }
  }, (0, _licensed_route_handler.createLicensedRouteHandler)(async (context, request, response) => {
    const {
      security: coreSecurity
    } = await context.core;
    return response.ok({
      body: coreSecurity.authc.getCurrentUser()
    });
  }));
  const basicParamsSchema = _configSchema.schema.object({
    username: _configSchema.schema.string({
      minLength: 1
    }),
    password: _configSchema.schema.string({
      minLength: 1
    })
  });
  function getLoginAttemptForProviderType(providerType, redirectURL, params) {
    if (providerType === _authentication.SAMLAuthenticationProvider.type) {
      return {
        type: _authentication.SAMLLogin.LoginInitiatedByUser,
        redirectURL
      };
    }
    if (providerType === _authentication.OIDCAuthenticationProvider.type) {
      return {
        type: _authentication.OIDCLogin.LoginInitiatedByUser,
        redirectURL
      };
    }
    if (providerType === _authentication.BasicAuthenticationProvider.type || providerType === _authentication.TokenAuthenticationProvider.type) {
      return params;
    }
    return undefined;
  }

  // Register the login route for serverless for the time being. Note: This route will move into the buildFlavor !== 'serverless' block below. See next line.
  // ToDo: In the serverless environment, we do not support API login - the only valid authentication type is SAML
  router.post({
    path: '/internal/security/login',
    security: {
      authz: {
        enabled: false,
        reason: `This route provides basic and token login capability, which is delegated to the internal authentication service`
      },
      authc: {
        enabled: false,
        reason: 'This route is used for authentication - it does not require existing authentication'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        providerType: _configSchema.schema.string(),
        providerName: _configSchema.schema.string(),
        currentURL: _configSchema.schema.string(),
        params: _configSchema.schema.conditional(_configSchema.schema.siblingRef('providerType'), _configSchema.schema.oneOf([_configSchema.schema.literal(_authentication.BasicAuthenticationProvider.type), _configSchema.schema.literal(_authentication.TokenAuthenticationProvider.type)]), basicParamsSchema, _configSchema.schema.never())
      })
    }
  }, (0, _licensed_route_handler.createLicensedRouteHandler)(async (context, request, response) => {
    const {
      providerType,
      providerName,
      currentURL,
      params
    } = request.body;
    const redirectURL = (0, _std.parseNextURL)(currentURL, basePath.serverBasePath);
    const authenticationResult = await getAuthenticationService().login(request, {
      provider: {
        name: providerName
      },
      redirectURL,
      value: getLoginAttemptForProviderType(providerType, redirectURL, params)
    });
    if (authenticationResult.redirected() || authenticationResult.succeeded()) {
      return response.ok({
        body: {
          location: authenticationResult.redirectURL || redirectURL
        },
        headers: authenticationResult.authResponseHeaders
      });
    }
    return response.unauthorized({
      body: authenticationResult.error,
      headers: authenticationResult.authResponseHeaders
    });
  }));
  if (buildFlavor !== 'serverless') {
    // In the serverless offering, the access agreement functionality isn't available.
    router.post({
      path: '/internal/security/access_agreement/acknowledge',
      security: {
        authz: {
          enabled: false,
          reason: `This route delegates authorization to the internal authentication service; there must be an authenticated user for this route to function`
        }
      },
      validate: false
    }, (0, _licensed_route_handler.createLicensedRouteHandler)(async (context, request, response) => {
      // If license doesn't allow access agreement we shouldn't handle request.
      if (!license.getFeatures().allowAccessAgreement) {
        logger.warn(`Attempted to acknowledge access agreement when license doesn't allow it.`);
        return response.forbidden({
          body: {
            message: `Current license doesn't support access agreement.`
          }
        });
      }
      await getAuthenticationService().acknowledgeAccessAgreement(request);
      return response.noContent();
    }));
  }
}