"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createBuiltinToolProvider = exports.createBuiltinProviderFn = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _definitions = require("../tool_types/definitions");
var _converter = require("./converter");
var _availability_cache = require("./availability_cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createBuiltinProviderFn = ({
  registry,
  toolTypes
}) => async ({
  request,
  space
}) => {
  return createBuiltinToolProvider({
    registry,
    toolTypes,
    request,
    space
  });
};
exports.createBuiltinProviderFn = createBuiltinProviderFn;
const createBuiltinToolProvider = ({
  registry,
  toolTypes,
  request,
  space
}) => {
  const definitionMap = toolTypes.filter(def => !(0, _definitions.isDisabledDefinition)(def)).reduce((map, def) => {
    map[def.toolType] = def;
    return map;
  }, {});
  const context = {
    spaceId: space,
    request
  };
  const availabilityCache = new _availability_cache.ToolAvailabilityCache();
  return {
    id: 'builtin',
    readonly: true,
    has(toolId) {
      return registry.has(toolId);
    },
    get(toolId) {
      const tool = registry.get(toolId);
      if (!tool) {
        throw (0, _onechatCommon.createToolNotFoundError)({
          toolId
        });
      }
      const definition = definitionMap[tool.type];
      if (!definition) {
        throw (0, _onechatCommon.createBadRequestError)(`Unknown type for tool '${toolId}': '${tool.type}'`);
      }
      return (0, _converter.convertTool)({
        tool,
        definition,
        context,
        cache: availabilityCache
      });
    },
    list() {
      const tools = registry.list();
      return tools.filter(tool => {
        // evict unknown tools - atm it's used for workflow tools if the plugin is disabled.
        return definitionMap[tool.type];
      }).map(tool => {
        const definition = definitionMap[tool.type];
        return (0, _converter.convertTool)({
          tool,
          definition,
          context,
          cache: availabilityCache
        });
      });
    }
  };
};
exports.createBuiltinToolProvider = createBuiltinToolProvider;