"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createModelProviderFactory = exports.createModelProvider = void 0;
var _inferenceCommon = require("@kbn/inference-common");
var _telemetry = require("../../telemetry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Utility function to creates a {@link ModelProviderFactoryFn}
 */
const createModelProviderFactory = factoryOpts => opts => {
  return createModelProvider({
    ...factoryOpts,
    ...opts
  });
};

/**
 * Utility function to create a {@link ModelProvider}
 */
exports.createModelProviderFactory = createModelProviderFactory;
const createModelProvider = ({
  inference,
  request,
  defaultConnectorId,
  trackingService
}) => {
  const getDefaultConnectorId = async () => {
    if (defaultConnectorId) {
      return defaultConnectorId;
    }
    const defaultConnector = await inference.getDefaultConnector(request);
    return defaultConnector.connectorId;
  };
  const completedCalls = [];
  const getUsageStats = () => {
    return {
      calls: completedCalls
    };
  };
  const getModel = async connectorId => {
    const completionCallback = event => {
      completedCalls.push({
        connectorId,
        tokens: event.tokens
      });
      if (trackingService && connector) {
        try {
          const provider = (0, _inferenceCommon.getConnectorProvider)(connector);
          const model = (0, _inferenceCommon.getConnectorModel)(connector);
          trackingService.trackLLMUsage(provider, model);
        } catch (e) {
          // ignore errors
        }
      }
    };
    const chatModel = await inference.getChatModel({
      request,
      connectorId,
      callbacks: {
        complete: [completionCallback]
      },
      chatModelOptions: {
        telemetryMetadata: _telemetry.MODEL_TELEMETRY_METADATA
      }
    });
    const inferenceClient = inference.getClient({
      request,
      bindTo: {
        connectorId
      },
      callbacks: {
        complete: [completionCallback]
      }
    });
    const connector = await inferenceClient.getConnectorById(connectorId);
    return {
      connector,
      chatModel,
      inferenceClient
    };
  };
  return {
    getDefaultModel: async () => getModel(await getDefaultConnectorId()),
    getModel: ({
      connectorId
    }) => getModel(connectorId),
    getUsageStats
  };
};
exports.createModelProvider = createModelProvider;