"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _agents = require("@kbn/onechat-common/agents");
var _onechatCommon = require("@kbn/onechat-common");
var _spaces = require("../../../../utils/spaces");
var _storage = require("./storage");
var _converters = require("./converters");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createClient = async ({
  space,
  request,
  elasticsearch,
  security,
  toolsService,
  logger
}) => {
  const authUser = security.authc.getCurrentUser(request);
  if (!authUser) {
    throw new Error('No user bound to the provided request');
  }
  const esClient = elasticsearch.client.asScoped(request).asInternalUser;
  const storage = (0, _storage.createStorage)({
    logger,
    esClient
  });
  const user = {
    id: authUser.profile_uid,
    username: authUser.username
  };
  return new AgentClientImpl({
    storage,
    user,
    request,
    space,
    toolsService
  });
};
exports.createClient = createClient;
class AgentClientImpl {
  constructor({
    storage,
    toolsService,
    user,
    request,
    space
  }) {
    (0, _defineProperty2.default)(this, "space", void 0);
    (0, _defineProperty2.default)(this, "request", void 0);
    (0, _defineProperty2.default)(this, "storage", void 0);
    (0, _defineProperty2.default)(this, "toolsService", void 0);
    (0, _defineProperty2.default)(this, "user", void 0);
    this.storage = storage;
    this.toolsService = toolsService;
    this.request = request;
    this.user = user;
    this.space = space;
  }
  async get(agentId) {
    const document = await this._get(agentId);
    if (!document) {
      throw (0, _onechatCommon.createAgentNotFoundError)({
        agentId
      });
    }
    if (!hasAccess({
      document,
      user: this.user
    })) {
      throw (0, _onechatCommon.createAgentNotFoundError)({
        agentId
      });
    }
    return (0, _converters.fromEs)(document);
  }
  async has(agentId) {
    const document = await this._get(agentId);
    return document !== undefined;
  }
  async list(options = {}) {
    const response = await this.storage.getClient().search({
      track_total_hits: false,
      size: 1000,
      query: {
        bool: {
          filter: [(0, _spaces.createSpaceDslFilter)(this.space)]
        }
      }
    });
    return response.hits.hits.map(hit => (0, _converters.fromEs)(hit));
  }
  async create(profile) {
    const now = new Date();
    const validationError = (0, _agents.validateAgentId)({
      agentId: profile.id,
      builtIn: false
    });
    if (validationError) {
      throw (0, _onechatCommon.createBadRequestError)(`Invalid agent id: "${profile.id}": ${validationError}`);
    }
    if (await this.exists(profile.id)) {
      throw (0, _onechatCommon.createBadRequestError)(`Agent with id ${profile.id} already exists.`);
    }
    await this.validateAgentToolSelection(profile.configuration.tools);
    const attributes = (0, _converters.createRequestToEs)({
      profile,
      space: this.space,
      creationDate: now
    });
    await this.storage.getClient().index({
      document: attributes
    });
    return this.get(profile.id);
  }
  async update(agentId, profileUpdate) {
    var _profileUpdate$config;
    const document = await this._get(agentId);
    if (!document) {
      throw (0, _onechatCommon.createAgentNotFoundError)({
        agentId
      });
    }
    if (!hasAccess({
      document,
      user: this.user
    })) {
      throw (0, _onechatCommon.createAgentNotFoundError)({
        agentId
      });
    }
    if ((_profileUpdate$config = profileUpdate.configuration) !== null && _profileUpdate$config !== void 0 && _profileUpdate$config.tools) {
      await this.validateAgentToolSelection(profileUpdate.configuration.tools);
    }
    const updatedConversation = (0, _converters.updateRequestToEs)({
      agentId,
      currentProps: document._source,
      update: profileUpdate,
      updateDate: new Date()
    });
    await this.storage.getClient().index({
      id: document._id,
      document: updatedConversation
    });
    return this.get(agentId);
  }
  async delete(options) {
    const {
      id
    } = options;
    const document = await this._get(id);
    if (!document) {
      throw (0, _onechatCommon.createAgentNotFoundError)({
        agentId: id
      });
    }
    if (!hasAccess({
      document,
      user: this.user
    })) {
      throw (0, _onechatCommon.createAgentNotFoundError)({
        agentId: id
      });
    }
    const deleteResponse = await this.storage.getClient().delete({
      id: document._id
    });
    return deleteResponse.result === 'deleted';
  }

  // Agent tool selection validation helper
  async validateAgentToolSelection(toolSelection) {
    const errors = await (0, _utils.validateToolSelection)({
      toolRegistry: await this.toolsService.getRegistry({
        request: this.request
      }),
      request: this.request,
      toolSelection
    });
    if (errors.length > 0) {
      throw (0, _onechatCommon.createBadRequestError)(`Agent tool selection validation failed:\n` + errors.map(e => `- ${e}`).join('\n'));
    }
  }
  async exists(agentId) {
    const document = await this._get(agentId);
    return !!document;
  }
  async _get(agentId) {
    const response = await this.storage.getClient().search({
      track_total_hits: false,
      size: 1,
      terminate_after: 1,
      query: {
        bool: {
          filter: [(0, _spaces.createSpaceDslFilter)(this.space), {
            bool: {
              // BWC compatibility with M1 - agentId was stored as the _id
              should: [{
                term: {
                  id: agentId
                }
              }, {
                term: {
                  _id: agentId
                }
              }],
              minimum_should_match: 1
            }
          }]
        }
      }
    });
    if (response.hits.hits.length === 0) {
      return undefined;
    } else {
      return response.hits.hits[0];
    }
  }
}
const hasAccess = ({
  document,
  user
}) => {
  // no access control for now
  return true;
};