"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerInternalToolsRoutes = registerInternalToolsRoutes;
var _configSchema = require("@kbn/config-schema");
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
var _wrap_handler = require("../wrap_handler");
var _features = require("../../../common/features");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerInternalToolsRoutes({
  router,
  coreSetup,
  getInternalServices,
  logger,
  pluginsSetup: {
    workflowsManagement
  }
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });

  // bulk delete tools
  router.post({
    path: `${_constants.internalApiPath}/tools/_bulk_delete`,
    validate: {
      body: _configSchema.schema.object({
        ids: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.manageOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      ids
    } = request.body;
    const {
      tools: toolService
    } = getInternalServices();
    const registry = await toolService.getRegistry({
      request
    });
    const deleteResults = await Promise.allSettled(ids.map(id => registry.delete(id)));
    const results = deleteResults.map((result, index) => {
      if (result.status !== 'fulfilled') {
        var _result$reason$toJSON, _result$reason$toJSON2, _result$reason;
        return {
          toolId: ids[index],
          success: false,
          reason: (_result$reason$toJSON = (_result$reason$toJSON2 = (_result$reason = result.reason).toJSON) === null || _result$reason$toJSON2 === void 0 ? void 0 : _result$reason$toJSON2.call(_result$reason)) !== null && _result$reason$toJSON !== void 0 ? _result$reason$toJSON : {
            error: {
              message: 'Unknown error'
            }
          }
        };
      }
      return {
        toolId: ids[index],
        success: true
      };
    });
    return response.ok({
      body: {
        results
      }
    });
  }));

  // resolve search sources (internal)
  router.get({
    path: `${_constants.internalApiPath}/tools/_resolve_search_sources`,
    validate: {
      query: _configSchema.schema.object({
        pattern: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const esClient = (await ctx.core).elasticsearch.client.asCurrentUser;
    const {
      pattern
    } = request.query;
    const {
      indices,
      aliases,
      data_streams: dataStreams
    } = await (0, _onechatGenaiUtils.listSearchSources)({
      pattern,
      includeHidden: false,
      includeKibanaIndices: false,
      excludeIndicesRepresentedAsAlias: true,
      excludeIndicesRepresentedAsDatastream: true,
      esClient
    });
    const results = [...indices.map(i => ({
      type: 'index',
      name: i.name
    })), ...aliases.map(a => ({
      type: 'alias',
      name: a.name
    })), ...dataStreams.map(d => ({
      type: 'data_stream',
      name: d.name
    }))];
    return response.ok({
      body: {
        results,
        total: results.length
      }
    });
  }));

  // list available tool types (internal)
  router.get({
    path: `${_constants.internalApiPath}/tools/_types_info`,
    validate: false,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      tools
    } = getInternalServices();
    const toolTypes = tools.getToolTypeInfo();
    return response.ok({
      body: {
        toolTypes
      }
    });
  }));

  // list workflows (internal)
  router.get({
    path: `${_constants.internalApiPath}/tools/_list_workflows`,
    validate: {
      query: _configSchema.schema.object({
        page: _configSchema.schema.number({
          defaultValue: 1
        }),
        limit: _configSchema.schema.number({
          defaultValue: 10000
        })
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    if (!workflowsManagement) {
      return response.ok({
        body: {
          results: []
        }
      });
    }
    const currentSpace = (await ctx.onechat).spaces.getSpaceId();
    const {
      results
    } = await workflowsManagement.management.getWorkflows({
      page: request.query.page,
      size: request.query.limit,
      enabled: [true]
    }, currentSpace);
    return response.ok({
      body: {
        results: results.map(workflow => ({
          id: workflow.id,
          name: workflow.name,
          description: workflow.description
        }))
      }
    });
  }));

  // get workflow (internal)
  router.get({
    path: `${_constants.internalApiPath}/tools/_get_workflow/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    if (!workflowsManagement) {
      return response.ok({
        body: {
          id: '',
          name: '',
          description: ''
        }
      });
    }
    const currentSpace = (await ctx.onechat).spaces.getSpaceId();
    const workflow = await workflowsManagement.management.getWorkflow(request.params.id, currentSpace);
    return response.ok({
      body: {
        id: workflow.id,
        name: workflow.name,
        description: workflow.description
      }
    });
  }));
}