"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useConversationScrollActions = void 0;
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DISTANCE_FROM_BOTTOM_THRESHOLD = 50; // pixels
const SCROLL_POSITION_CHECK_INTERVAL = 1500; // milliseconds
const DEBOUNCE_DELAY = 20; // milliseconds

const scrollToMostRecentRound = ({
  scrollContainer,
  position,
  scrollBehavior = 'smooth'
}) => {
  requestAnimationFrame(() => {
    // Find the rounds container within the specific scroll container
    const conversationRoundsElement = scrollContainer.querySelector('[id="onechatConversationRoundsContainer"]');
    if (conversationRoundsElement) {
      const rounds = conversationRoundsElement.children;
      if (rounds.length >= 1) {
        // Get the last round (the user's last message)
        const lastRound = rounds[rounds.length - 1];
        lastRound.scrollIntoView({
          behavior: scrollBehavior,
          block: position
        });
      }
    }
  });
};
const checkScrollPosition = (scrollContainer, setShowScrollButton) => {
  const {
    scrollTop,
    scrollHeight,
    clientHeight
  } = scrollContainer;
  const distanceFromBottom = scrollHeight - (scrollTop + clientHeight);
  setShowScrollButton(distanceFromBottom > DISTANCE_FROM_BOTTOM_THRESHOLD);
};
const createDebouncedCheckScrollPosition = (scrollContainer, setShowScrollButton) => {
  let debounceTimeout;
  return () => {
    clearTimeout(debounceTimeout);
    debounceTimeout = setTimeout(() => checkScrollPosition(scrollContainer, setShowScrollButton), DEBOUNCE_DELAY);
  };
};
const useConversationScrollActions = ({
  isResponseLoading,
  conversationId,
  scrollContainer
}) => {
  const [showScrollButton, setShowScrollButton] = (0, _react.useState)(false);
  (0, _react.useEffect)(() => {
    if (!scrollContainer) return;
    const debouncedCheckScrollPosition = createDebouncedCheckScrollPosition(scrollContainer, setShowScrollButton);
    scrollContainer.addEventListener('scroll', debouncedCheckScrollPosition);

    // Set up interval for streaming check (only when response is loading)
    let interval;
    if (isResponseLoading) {
      interval = setInterval(() => checkScrollPosition(scrollContainer, setShowScrollButton), SCROLL_POSITION_CHECK_INTERVAL);
    } else {
      // when the content stops streaming, check the scroll position
      checkScrollPosition(scrollContainer, setShowScrollButton);
    }
    return () => {
      scrollContainer.removeEventListener('scroll', debouncedCheckScrollPosition);
      if (interval) clearInterval(interval);
    };
  }, [isResponseLoading, conversationId, scrollContainer]);
  const stickToBottom = (0, _react.useCallback)(() => {
    if (!scrollContainer) {
      return;
    }
    scrollContainer.scrollTop = scrollContainer.scrollHeight;
  }, [scrollContainer]);

  // Scrolls the most recent round to the top of it's parent scroll container
  const scrollToMostRecentRoundTop = (0, _react.useCallback)(() => {
    if (!scrollContainer) return;
    scrollToMostRecentRound({
      scrollContainer,
      position: 'start'
    });
  }, [scrollContainer]);

  // Scrolls the most recent round to the bottom of it's parent scroll container
  const scrollToMostRecentRoundBottom = (0, _react.useCallback)(() => {
    if (!scrollContainer) return;
    scrollToMostRecentRound({
      scrollContainer,
      position: 'end'
    });
  }, [scrollContainer]);
  return {
    showScrollButton,
    scrollToMostRecentRoundTop,
    scrollToMostRecentRoundBottom,
    stickToBottom
  };
};
exports.useConversationScrollActions = useConversationScrollActions;