"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createNlToEsqlGraph = void 0;
var _zod = require("@kbn/zod");
var _langgraph = require("@langchain/langgraph");
var _common = require("@kbn/inference-plugin/common");
var _messages = require("../../langchain/messages");
var _esql = require("../utils/esql");
var _resources = require("../utils/resources");
var _prompts = require("./prompts");
var _actions = require("./actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const StateAnnotation = _langgraph.Annotation.Root({
  // inputs
  nlQuery: (0, _langgraph.Annotation)(),
  target: (0, _langgraph.Annotation)(),
  executeQuery: (0, _langgraph.Annotation)(),
  maxRetries: (0, _langgraph.Annotation)(),
  additionalInstructions: (0, _langgraph.Annotation)(),
  additionalContext: (0, _langgraph.Annotation)(),
  rowLimit: (0, _langgraph.Annotation)(),
  // internal
  resource: (0, _langgraph.Annotation)(),
  currentTry: (0, _langgraph.Annotation)({
    reducer: (a, b) => b,
    default: () => 0
  }),
  actions: (0, _langgraph.Annotation)({
    reducer: (a, b) => [...a, ...b],
    default: () => []
  }),
  // outputs
  answer: (0, _langgraph.Annotation)(),
  query: (0, _langgraph.Annotation)(),
  results: (0, _langgraph.Annotation)(),
  error: (0, _langgraph.Annotation)()
});
const createNlToEsqlGraph = ({
  model,
  esClient,
  docBase,
  logger,
  events
}) => {
  // resolve the search target / generate sampling data
  const resolveTarget = async state => {
    const resolvedResource = await (0, _resources.resolveResourceWithSamplingStats)({
      resourceName: state.target,
      samplingSize: 100,
      esClient
    });
    return {
      resource: resolvedResource
    };
  };

  // request doc step - retrieve the list of relevant commands and functions that may be useful to generate the query
  const requestDocumentation = async state => {
    const requestDocModel = model.chatModel.withStructuredOutput(_zod.z.object({
      commands: _zod.z.array(_zod.z.string()).optional().describe('ES|QL source and processing commands to get documentation for.'),
      functions: _zod.z.array(_zod.z.string()).optional().describe('ES|QL functions to get documentation for.')
    }).describe('Tool to use to request ES|QL documentation'), {
      name: 'request_documentation'
    });
    const {
      commands = [],
      functions = []
    } = await requestDocModel.invoke((0, _prompts.createRequestDocumentationPrompt)({
      nlQuery: state.nlQuery,
      prompts: docBase.getPrompts(),
      resource: state.resource
    }));
    const requestedKeywords = [...commands, ...functions];
    const fetchedDoc = docBase.getDocumentation(requestedKeywords);
    const action = {
      type: 'request_documentation',
      requestedKeywords,
      fetchedDoc
    };
    return {
      actions: [action]
    };
  };

  // generate esql step - generate the esql query based on the doc and the user's input
  const generateEsql = async state => {
    const generateModel = model.chatModel;
    const response = await generateModel.invoke((0, _prompts.createGenerateEsqlPrompt)({
      nlQuery: state.nlQuery,
      prompts: docBase.getPrompts(),
      resource: state.resource,
      previousActions: state.actions,
      additionalInstructions: state.additionalInstructions,
      additionalContext: state.additionalContext,
      rowLimit: state.rowLimit
    }));
    const responseText = (0, _messages.extractTextContent)(response);
    const queries = (0, _esql.extractEsqlQueries)(responseText);
    const action = {
      type: 'generate_query',
      success: queries.length > 0,
      query: queries[0],
      response: responseText
    };
    return {
      actions: [action],
      currentTry: state.currentTry + 1
    };
  };
  const branchAfterGenerate = async state => {
    const lastAction = state.actions[state.actions.length - 1];
    if (!(0, _actions.isGenerateQueryAction)(lastAction)) {
      throw new Error(`Last action is not a generate_query action`);
    }
    if (lastAction.success) {
      return 'autocorrect_query';
    } else if (state.currentTry >= state.maxRetries) {
      return 'finalize';
    } else {
      return 'generate_esql';
    }
  };

  // autocorrect step - try to correct common mistakes in the esql query
  const autocorrectQuery = async state => {
    const lastAction = state.actions[state.actions.length - 1];
    if (!(0, _actions.isGenerateQueryAction)(lastAction) || !lastAction.query) {
      throw new Error(`Last action is not a generate_query action`);
    }
    const correction = (0, _common.correctCommonEsqlMistakes)(lastAction.query);
    const action = {
      type: 'autocorrect_query',
      wasCorrected: correction.isCorrection,
      input: correction.input,
      output: correction.output
    };
    return {
      actions: [action]
    };
  };
  const branchAfterAutocorrect = async state => {
    if (state.executeQuery) {
      return 'execute_query';
    } else {
      return 'finalize';
    }
  };

  // execute query step - execute the query and get the results
  const executeQuery = async state => {
    let query;
    const lastAction = state.actions[state.actions.length - 1];
    if ((0, _actions.isGenerateQueryAction)(lastAction) && lastAction.query) {
      query = lastAction.query;
    } else if ((0, _actions.isAutocorrectQueryAction)(lastAction)) {
      query = lastAction.output;
    } else {
      throw new Error(`Last action is not a generate_query or autocorrect_query action`);
    }
    let action;
    try {
      const results = await (0, _esql.executeEsql)({
        query,
        esClient
      });
      action = {
        type: 'execute_query',
        success: true,
        query,
        results
      };
    } catch (e) {
      action = {
        type: 'execute_query',
        success: false,
        query,
        error: e.message
      };
    }
    return {
      actions: [action]
    };
  };
  const branchAfterQueryExecution = async state => {
    const lastAction = state.actions[state.actions.length - 1];
    if (!(0, _actions.isExecuteQueryAction)(lastAction)) {
      throw new Error(`Last action is not an execute_query action`);
    }
    if (lastAction.success || state.currentTry >= state.maxRetries) {
      return 'finalize';
    } else {
      return 'generate_esql';
    }
  };

  // finalize step - process / generate the outputs
  const finalize = async state => {
    const lastAction = state.actions[state.actions.length - 1];
    const generateActions = state.actions.filter(_actions.isGenerateQueryAction);

    // ended via query execution - either successful or failure hitting max retries
    if ((0, _actions.isExecuteQueryAction)(lastAction)) {
      return {
        answer: generateActions[generateActions.length - 1].response,
        query: lastAction.query,
        results: lastAction.results,
        error: lastAction.error
      };
    }
    // ended via autocorrect - if executeQuery=false
    if ((0, _actions.isAutocorrectQueryAction)(lastAction)) {
      return {
        answer: generateActions[generateActions.length - 1].response,
        query: lastAction.output
      };
    }
    // ended via query generation - because the LLM didn't generate a query for some reason
    if ((0, _actions.isGenerateQueryAction)(lastAction)) {
      return {
        error: 'No query was generated',
        answer: lastAction.response,
        query: lastAction.query
      };
    }

    // can't really happen, but just to make TS happy
    return {};
  };
  const graph = new _langgraph.StateGraph(StateAnnotation)
  // nodes
  .addNode('resolve_target', resolveTarget).addNode('request_documentation', requestDocumentation).addNode('generate_esql', generateEsql).addNode('autocorrect_query', autocorrectQuery).addNode('execute_query', executeQuery).addNode('finalize', finalize)
  // edges
  .addEdge('__start__', 'resolve_target').addEdge('resolve_target', 'request_documentation').addEdge('request_documentation', 'generate_esql').addConditionalEdges('generate_esql', branchAfterGenerate, {
    generate_esql: 'generate_esql',
    autocorrect_query: 'autocorrect_query',
    finalize: 'finalize'
  }).addConditionalEdges('autocorrect_query', branchAfterAutocorrect, {
    execute_query: 'execute_query',
    finalize: 'finalize'
  }).addConditionalEdges('execute_query', branchAfterQueryExecution, {
    generate_esql: 'generate_esql',
    finalize: 'finalize'
  }).addEdge('finalize', '__end__').compile();
  return graph;
};
exports.createNlToEsqlGraph = createNlToEsqlGraph;