"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RelatedDashboardsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _common = require("@kbn/dashboard-plugin/common");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RelatedDashboardsClient {
  constructor(logger, getDashboard, scanDashboards, alertsClient, alertId, referencedPanelManager) {
    (0, _defineProperty2.default)(this, "dashboardsById", new Map());
    (0, _defineProperty2.default)(this, "alert", null);
    (0, _defineProperty2.default)(this, "getPanelIndicesMap", {
      lens: panel => {
        var _references;
        let references = this.isLensVizAttributes(panel.config) ? panel.config.attributes.references : undefined;
        if (!references && panel.uid) {
          var _this$referencedPanel;
          references = (_this$referencedPanel = this.referencedPanelManager.getByUid(panel.uid)) === null || _this$referencedPanel === void 0 ? void 0 : _this$referencedPanel.references;
        }
        if ((_references = references) !== null && _references !== void 0 && _references.length) {
          return new Set(references.filter(r => r.name.match(`indexpattern`)).map(reference => reference.id));
        }
      }
    });
    (0, _defineProperty2.default)(this, "getPanelFieldsMap", {
      lens: panel => {
        let state = this.isLensVizAttributes(panel.config) ? panel.config.attributes.state : undefined;
        if (!state && panel.uid) {
          var _this$referencedPanel2;
          state = (_this$referencedPanel2 = this.referencedPanelManager.getByUid(panel.uid)) === null || _this$referencedPanel2 === void 0 ? void 0 : _this$referencedPanel2.state;
        }
        if (this.isLensAttributesState(state)) {
          var _state$datasourceStat;
          const fields = new Set();
          const dataSourceLayers = ((_state$datasourceStat = state.datasourceStates.formBased) === null || _state$datasourceStat === void 0 ? void 0 : _state$datasourceStat.layers) || {};
          Object.values(dataSourceLayers).forEach(ds => {
            const columns = ds.columns;
            Object.values(columns).forEach(col => {
              if (this.hasSourceField(col)) {
                fields.add(col.sourceField);
              }
            });
          });
          return fields;
        }
      }
    });
    this.logger = logger;
    this.getDashboard = getDashboard;
    this.scanDashboards = scanDashboards;
    this.alertsClient = alertsClient;
    this.alertId = alertId;
    this.referencedPanelManager = referencedPanelManager;
  }
  async fetchRelatedDashboards() {
    const [alertDocument] = await Promise.all([this.alertsClient.getAlertById(this.alertId), this.fetchFirst500Dashboards()]);
    this.setAlert(alertDocument);
    const [suggestedDashboards, linkedDashboards] = await Promise.all([this.fetchSuggestedDashboards(), this.getLinkedDashboards()]);
    const filteredSuggestedDashboards = suggestedDashboards.filter(suggested => !linkedDashboards.some(linked => linked.id === suggested.id));
    return {
      suggestedDashboards: filteredSuggestedDashboards.slice(0, 10),
      // limit to 10 suggested dashboards
      linkedDashboards
    };
  }
  hasSourceField(c) {
    return 'sourceField' in c;
  }
  isLensAttributesState(state) {
    return typeof state === 'object' && state !== null && 'datasourceStates' in state;
  }
  setAlert(alert) {
    this.alert = alert;
  }
  checkAlert() {
    if (!this.alert) throw new Error(`Alert with id ${this.alertId} not found. Could not fetch related dashboards.`);
    return this.alert;
  }
  async fetchSuggestedDashboards() {
    const alert = this.checkAlert();
    if (!(0, _helpers.isSuggestedDashboardsValidRuleTypeId)(alert.getRuleTypeId())) return [];
    const allSuggestedDashboards = new Set();
    const relevantDashboardsById = new Map();
    const index = this.getRuleQueryIndex();
    const allRelevantFields = alert.getAllRelevantFields();
    if (index) {
      const {
        dashboards
      } = this.getDashboardsByIndex(index);
      dashboards.forEach(dashboard => allSuggestedDashboards.add(dashboard));
    }
    if (allRelevantFields.length > 0) {
      const {
        dashboards
      } = this.getDashboardsByField(allRelevantFields);
      dashboards.forEach(dashboard => allSuggestedDashboards.add(dashboard));
    }
    allSuggestedDashboards.forEach(dashboard => {
      var _relevantDashboardsBy;
      relevantDashboardsById.set(dashboard.id, {
        ...dashboard,
        matchedBy: {
          ...((_relevantDashboardsBy = relevantDashboardsById.get(dashboard.id)) === null || _relevantDashboardsBy === void 0 ? void 0 : _relevantDashboardsBy.matchedBy),
          ...dashboard.matchedBy
        },
        score: this.getScore(dashboard)
      });
    });
    const sortedDashboards = Array.from(relevantDashboardsById.values()).sort((a, b) => {
      return b.score - a.score;
    });
    return sortedDashboards;
  }
  async fetchDashboards({
    page,
    perPage = 20,
    limit
  }) {
    const results = await this.scanDashboards(page, perPage);
    for (const dashboard of results.dashboards) {
      for (const panel of dashboard.panels) {
        if ((0, _common.isDashboardPanel)(panel) && (0, _helpers.isSuggestedDashboardsValidPanelType)(panel.type) && ((0, _lodash.isEmpty)(panel.config) || !panel.config.attributes)) {
          this.referencedPanelManager.addReferencedPanel({
            dashboardId: dashboard.id,
            references: dashboard.references,
            panel
          });
        }
      }
      this.dashboardsById.set(dashboard.id, dashboard);
    }
    await this.referencedPanelManager.fetchReferencedPanels();
    const fetchedUntil = (page - 1) * perPage + results.dashboards.length;
    if (results.total <= fetchedUntil) {
      return;
    }
    if (limit && fetchedUntil >= limit) {
      return;
    }
    await this.fetchDashboards({
      page: page + 1,
      perPage,
      limit
    });
  }
  async fetchFirst500Dashboards() {
    await this.fetchDashboards({
      page: 1,
      perPage: 500,
      limit: 500
    });
  }
  getDashboardsByIndex(index) {
    const relevantDashboards = [];
    this.dashboardsById.forEach((d, id) => {
      const panels = d.panels.filter(_common.isDashboardPanel);
      const matchingPanels = this.getPanelsByIndex(index, panels);
      if (matchingPanels.length > 0) {
        this.logger.debug(() => `Found ${matchingPanels.length} panel(s) in dashboard ${id} using index ${index}`);
        relevantDashboards.push({
          id,
          title: d.title,
          description: d.description,
          tags: d.tags,
          matchedBy: {
            index: [index]
          },
          score: 0 // scores are computed when dashboards are deduplicated
        });
      }
    });
    return {
      dashboards: relevantDashboards
    };
  }
  getDashboardsByField(fields) {
    const relevantDashboards = [];
    this.dashboardsById.forEach((d, id) => {
      const panels = d.panels.filter(_common.isDashboardPanel);
      const matchingPanels = this.getPanelsByField(fields, panels);
      const allMatchingFields = new Set(matchingPanels.map(p => Array.from(p.matchingFields)).flat());
      if (matchingPanels.length > 0) {
        this.logger.debug(() => `Found ${matchingPanels.length} panel(s) in dashboard ${id} using field(s) ${Array.from(allMatchingFields).toString()}`);
        relevantDashboards.push({
          id,
          title: d.title,
          description: d.description,
          tags: d.tags,
          matchedBy: {
            fields: Array.from(allMatchingFields)
          },
          score: 0 // scores are computed when dashboards are deduplicated
        });
      }
    });
    return {
      dashboards: relevantDashboards
    };
  }
  getPanelsByIndex(index, panels) {
    const panelsByIndex = panels.filter(p => this.getPanelIndices(p).has(index));
    return panelsByIndex;
  }
  getPanelsByField(fields, panels) {
    const panelsByField = panels.reduce((acc, p) => {
      const matchingFields = fields.filter(f => this.getPanelFields(p).has(f));
      if (matchingFields.length) {
        acc.push({
          matchingFields: new Set(matchingFields),
          panel: p
        });
      }
      return acc;
    }, []);
    return panelsByField;
  }
  getPanelIndices(panel) {
    const indices = (0, _helpers.isSuggestedDashboardsValidPanelType)(panel.type) ? this.getPanelIndicesMap[panel.type](panel) : undefined;
    return indices !== null && indices !== void 0 ? indices : new Set();
  }
  getPanelFields(panel) {
    const fields = (0, _helpers.isSuggestedDashboardsValidPanelType)(panel.type) ? this.getPanelFieldsMap[panel.type](panel) : undefined;
    return fields !== null && fields !== void 0 ? fields : new Set();
  }
  isLensVizAttributes(config) {
    const {
      attributes
    } = config !== null && config !== void 0 ? config : {};
    if (!attributes) {
      return false;
    }
    return Boolean(attributes) && typeof attributes === 'object' && 'type' in attributes && attributes.type === 'lens';
  }
  getRuleQueryIndex() {
    const alert = this.checkAlert();
    const index = alert.getRuleQueryIndex();
    return index;
  }
  async getLinkedDashboards() {
    var _rule$artifacts;
    const alert = this.checkAlert();
    const ruleId = alert.getRuleId();
    if (!ruleId) {
      throw new Error(`Alert with id ${this.alertId} does not have a rule ID. Could not fetch linked dashboards.`);
    }
    const rule = await this.alertsClient.getRuleById(ruleId);
    if (!rule) {
      throw new Error(`Rule with id ${ruleId} not found. Could not fetch linked dashboards for alert with id ${this.alertId}.`);
    }
    const linkedDashboardsArtifacts = ((_rule$artifacts = rule.artifacts) === null || _rule$artifacts === void 0 ? void 0 : _rule$artifacts.dashboards) || [];
    const linkedDashboards = await this.getLinkedDashboardsByIds(linkedDashboardsArtifacts.map(d => d.id));
    return linkedDashboards;
  }
  async getLinkedDashboardsByIds(ids) {
    const linkedDashboardsResponse = await Promise.all(ids.map(id => this.getLinkedDashboardById(id)));
    return linkedDashboardsResponse.filter(dashboard => Boolean(dashboard));
  }
  async getLinkedDashboardById(id) {
    try {
      const dashboard = await this.getDashboard(id);
      return {
        ...dashboard,
        matchedBy: {
          linked: true
        }
      };
    } catch (error) {
      if (error.output.statusCode === 404) {
        this.logger.warn(`Linked dashboard with id ${id} not found. Skipping.`);
        return null;
      }
      throw new Error(`Error fetching dashboard with id ${id}: ${error.message || error}`);
    }
  }
  getMatchingFields(dashboard) {
    const matchingFields = new Set();
    // grab all the top level arrays from the matchedBy object via Object.values
    Object.values((0, _lodash.omit)(dashboard.matchedBy, 'linked')).forEach(match => {
      // add the values of each array to the matchingFields set
      match.forEach(value => {
        matchingFields.add(value);
      });
    });
    return Array.from(matchingFields);
  }
  getScore(dashboard) {
    const alert = this.checkAlert();
    const allRelevantFields = alert.getAllRelevantFields();
    const index = this.getRuleQueryIndex();
    const setA = new Set([...allRelevantFields, ...(index ? [index] : [])]);
    const setB = new Set(this.getMatchingFields(dashboard));
    const intersection = new Set([...setA].filter(item => setB.has(item)));
    const union = new Set([...setA, ...setB]);
    return intersection.size / union.size;
  }
}
exports.RelatedDashboardsClient = RelatedDashboardsClient;