"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchDashboards = exports.fetchDashboard = void 0;
var _lodash = require("lodash");
var _kibana_services = require("../../services/kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getDashboardItem(id, dashboardState) {
  return {
    id,
    title: dashboardState.title,
    ...(dashboardState.description && {
      description: dashboardState.description
    })
  };
}

/**
 * ----------------------------------
 * Fetch a single dashboard
 * ----------------------------------
 */

const fetchDashboard = async dashboardId => {
  const findDashboardsService = await _kibana_services.dashboardServices.findDashboardsService();
  const response = await findDashboardsService.findById(dashboardId);
  if (response.status === 'error') {
    throw new Error(response.error.message);
  }
  return getDashboardItem(response.id, response.attributes);
};

/**
 * ----------------------------------
 * Fetch lists of dashboards
 * ----------------------------------
 */
exports.fetchDashboard = fetchDashboard;
const fetchDashboards = async ({
  search = '',
  size = 10,
  parentDashboardId,
  selectedDashboardId
}) => {
  const findDashboardsService = await _kibana_services.dashboardServices.findDashboardsService();
  const responses = await findDashboardsService.search({
    search,
    per_page: size
  });
  let dashboardList = responses.dashboards.map(({
    id,
    data
  }) => {
    return getDashboardItem(id, data);
  });

  /** If there is no search string... */
  if ((0, _lodash.isEmpty)(search)) {
    /** ... filter out both the parent and selected dashboard from the list ... */
    dashboardList = (0, _lodash.filter)(dashboardList, dash => {
      return dash.id !== parentDashboardId && dash.id !== selectedDashboardId;
    });

    /** ... so that we can force them to the top of the list as necessary. */
    if (parentDashboardId) {
      dashboardList.unshift(await fetchDashboard(parentDashboardId));
    }
    if (selectedDashboardId && selectedDashboardId !== parentDashboardId) {
      const selectedDashboard = await fetchDashboard(selectedDashboardId).catch(() => {
        /**
         * Swallow the error thrown, since this just means the selected dashboard was deleted and therefore
         * it should not be added to the top of the dashboard list
         */
      });
      if (selectedDashboard) dashboardList.unshift(await fetchDashboard(selectedDashboardId));
    }
  }
  return dashboardList;
};
exports.fetchDashboards = fetchDashboards;