"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToRawColorMappings = convertToRawColorMappings;
exports.isDeprecatedColorMapping = isDeprecatedColorMapping;
var _common = require("@kbn/data-plugin/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Converts old stringified colorMapping configs to new raw value configs
 *
 * Also fixes loop mode issue https://github.com/elastic/kibana/issues/231165
 */
function convertToRawColorMappings({
  ...colorMapping
}, columnMeta) {
  const isLegacyLoopMode = 'assignmentMode' in colorMapping && colorMapping.assignmentMode === 'auto' || colorMapping.assignments.length === 0;
  delete colorMapping.assignmentMode;
  return {
    ...colorMapping,
    assignments: colorMapping.assignments.map(oldAssignment => {
      if (isValidColorMappingAssignment(oldAssignment)) return oldAssignment;
      return convertColorMappingAssignment(oldAssignment, columnMeta);
    }),
    specialAssignments: colorMapping.specialAssignments.map((oldAssignment, i) => {
      const isBadColor = isLegacyLoopMode && i === 0;
      const newColor = isBadColor ? {
        type: 'loop'
      } : oldAssignment.color;
      if (isValidColorMappingAssignment(oldAssignment)) {
        return {
          ...oldAssignment,
          color: newColor
        };
      }
      return {
        color: newColor,
        touched: oldAssignment.touched,
        rules: [oldAssignment.rule]
      };
    })
  };
}
function convertColorMappingAssignment(oldAssignment, columnMeta) {
  return {
    color: oldAssignment.color,
    touched: oldAssignment.touched,
    rules: convertColorMappingRule(oldAssignment.rule, columnMeta)
  };
}
const NO_VALUE = Symbol('no-value');
function convertColorMappingRule(rule, columnMeta) {
  switch (rule.type) {
    case 'auto':
      return [];
    case 'matchExactly':
      return rule.values.map(value => {
        const rawValue = convertToRawValue(value, columnMeta);
        if (rawValue !== NO_VALUE) {
          return {
            type: 'raw',
            value: rawValue
          };
        }
        return {
          type: 'match',
          pattern: String(value),
          matchEntireWord: true,
          matchCase: true
        };
      });

    // Rules below not yet used, adding conversions for completeness
    case 'matchExactlyCI':
      return rule.values.map(value => ({
        type: 'match',
        pattern: Array.isArray(value) ? value.join(' ') : value,
        matchEntireWord: true,
        matchCase: false
      }));
    case 'regex':
      return [{
        type: rule.type,
        pattern: rule.values
      }];
    case 'range':
    default:
      return [rule];
  }
}

/**
 * Attempts to convert the previously stringified raw values into their raw/serialized form
 *
 * Note: we use the `NO_VALUE` symbol to avoid collisions with falsy raw values
 */
function convertToRawValue(value, columnMeta) {
  if (!columnMeta) return NO_VALUE;

  // all array values are multi-term
  if (columnMeta.fieldType === 'multi_terms' || Array.isArray(value)) {
    if (typeof value === 'string') return NO_VALUE; // cannot assume this as multi-field
    return new _common.MultiFieldKey({
      key: value
    }).serialize();
  }
  if (columnMeta.fieldType === 'range') {
    return _common.RangeKey.isRangeKeyString(value) ? _common.RangeKey.fromString(value).serialize() : NO_VALUE;
  }
  switch (columnMeta.dataType) {
    case 'boolean':
      if (value === '__other__' || value === 'true' || value === 'false') return value; // bool could have __other__ as a string
      if (value === '0' || value === '1') return Number(value);
      break;
    case 'number':
    case 'date':
      if (value === '__other__') return value; // numbers can have __other__ as a string
      const numberValue = Number(value);
      if (isFinite(numberValue)) return numberValue;
      break;
    case 'string':
    case 'ip':
      return value;
    // unable to distinguish manually added values
    default:
      return NO_VALUE;
    // treat all other other dataType as custom match string values
  }
  return NO_VALUE;
}
function isValidColorMappingAssignment(assignment) {
  return 'rules' in assignment;
}
function isDeprecatedColorMapping(colorMapping) {
  if (!colorMapping) return false;
  return Boolean(colorMapping.assignments && (colorMapping.assignments.some(assignment => !isValidColorMappingAssignment(assignment)) || colorMapping.specialAssignments.some(specialAssignment => !isValidColorMappingAssignment(specialAssignment))));
}