"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAgentPolicyWithPackages = createAgentPolicyWithPackages;
var _agent_policies_helpers = require("../../common/services/agent_policies_helpers");
var _common = require("../../common");
var _2 = require(".");
var _package_policies = require("./package_policies");
var _packages = require("./epm/packages");
var _api_keys = require("./api_keys");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getFleetServerAgentPolicyId(soClient, agentPolicyService) {
  const logger = _2.appContextService.getLogger().get('getFleetServerAgentPolicyId');
  logger.debug(`Retrieving fleet server agent policy id using soClient scoped to [${soClient.getCurrentNamespace()}]`);
  let agentPolicyId;
  // creating first fleet server policy with id '(space-)?fleet-server-policy'
  let agentPolicy;
  try {
    agentPolicy = await agentPolicyService.get(soClient, (0, _agent_policies_helpers.getDefaultFleetServerpolicyId)(soClient.getCurrentNamespace()), false);
  } catch (err) {
    if (!err.isBoom || err.output.statusCode !== 404) {
      throw err;
    }
  }
  if (!agentPolicy) {
    agentPolicyId = (0, _agent_policies_helpers.getDefaultFleetServerpolicyId)(soClient.getCurrentNamespace());
  }
  logger.debug(`Returning agent policy id [${agentPolicyId}]`);
  return agentPolicyId;
}
async function createPackagePolicy(soClient, esClient, agentPolicyService, agentPolicy, packageToInstall, options) {
  var _agentPolicy$space_id;
  const newPackagePolicy = await _2.packagePolicyService.buildPackagePolicyFromPackage(soClient, packageToInstall).catch(async error => {
    // rollback agent policy on error
    await agentPolicyService.delete(soClient, esClient, agentPolicy.id, {
      force: true,
      user: options.user
    });
    throw error;
  });
  if (!newPackagePolicy) return;
  newPackagePolicy.policy_id = agentPolicy.id;
  newPackagePolicy.policy_ids = [agentPolicy.id];
  newPackagePolicy.name = await (0, _package_policies.incrementPackageName)(soClient, packageToInstall, (_agentPolicy$space_id = agentPolicy.space_ids) !== null && _agentPolicy$space_id !== void 0 ? _agentPolicy$space_id : [options.spaceId]);
  if (agentPolicy.supports_agentless) {
    newPackagePolicy.supports_agentless = agentPolicy.supports_agentless;
  }
  await _2.packagePolicyService.create(soClient, esClient, newPackagePolicy, {
    spaceId: options.spaceId,
    user: options.user,
    bumpRevision: false,
    authorizationHeader: options.authorizationHeader,
    force: options.force
  });
}
async function createAgentPolicyWithPackages({
  soClient,
  esClient,
  agentPolicyService,
  newPolicy,
  hasFleetServer,
  withSysMonitoring: withSysMonitoringParams,
  monitoringEnabled: monitoringEnabledParams,
  spaceId,
  user,
  authorizationHeader,
  force,
  forcePackagePolicyCreation
}) {
  const logger = _2.appContextService.getLogger().get('createAgentPolicyWithPackages');
  logger.debug(`creating policy [${newPolicy.name}] for space [${spaceId}] using soClient scoped to [${soClient.getCurrentNamespace()}]`);
  let agentPolicyId = newPolicy.id;
  const packagesToInstall = [];
  if (hasFleetServer) {
    packagesToInstall.push(_common.FLEET_SERVER_PACKAGE);
    agentPolicyId = agentPolicyId || (await getFleetServerAgentPolicyId(soClient, agentPolicyService));
    if (agentPolicyId === (0, _agent_policies_helpers.getDefaultFleetServerpolicyId)(spaceId)) {
      // setting first fleet server policy to default, so that fleet server can enroll without setting policy_id
      newPolicy.is_default_fleet_server = true;
    }
  }
  const withSysMonitoring = withSysMonitoringParams && !newPolicy.supports_agentless;
  if (!withSysMonitoring && withSysMonitoringParams) {
    logger.info(`Disabling system monitoring for agentless policy [${newPolicy.name}]`);
  }
  const monitoringEnabled = newPolicy.supports_agentless && monitoringEnabledParams !== null && monitoringEnabledParams !== void 0 && monitoringEnabledParams.length ? [] : monitoringEnabledParams;
  if (monitoringEnabledParams !== null && monitoringEnabledParams !== void 0 && monitoringEnabledParams.length && !(monitoringEnabled !== null && monitoringEnabled !== void 0 && monitoringEnabled.length)) {
    logger.info(`Disabling monitoring for agentless policy [${newPolicy.name}]`);
  }
  if (withSysMonitoring) {
    packagesToInstall.push(_common.FLEET_SYSTEM_PACKAGE);
  }
  if (monitoringEnabled !== null && monitoringEnabled !== void 0 && monitoringEnabled.length) {
    packagesToInstall.push(_common.FLEET_ELASTIC_AGENT_PACKAGE);
  }
  if (packagesToInstall.length > 0) {
    logger.debug(() => `Installing packages [${packagesToInstall.join(', ')}]`);
    await (0, _packages.bulkInstallPackages)({
      savedObjectsClient: soClient,
      esClient,
      packagesToInstall,
      spaceId,
      authorizationHeader,
      force
    });
  }
  const {
    id,
    monitoring_enabled: _,
    ...policy
  } = newPolicy; // omit id from create object

  const agentPolicy = await agentPolicyService.create(soClient, esClient, {
    ...policy,
    monitoring_enabled: monitoringEnabled
  }, {
    user,
    id: agentPolicyId,
    authorizationHeader,
    hasFleetServer,
    skipDeploy: true // skip deploying the policy until package policies are added
  });

  // Since agentPolicyService does not handle multispace assignments, we need to keep this context with package policy creation
  const agentPolicyWithStagedSpaces = {
    ...agentPolicy,
    space_ids: newPolicy.space_ids
  };

  // Create the fleet server package policy and add it to agent policy.
  if (hasFleetServer) {
    await createPackagePolicy(soClient, esClient, agentPolicyService, agentPolicyWithStagedSpaces, _common.FLEET_SERVER_PACKAGE, {
      spaceId,
      user,
      authorizationHeader,
      force: force || forcePackagePolicyCreation
    });
  }

  // Create the system monitoring package policy and add it to agent policy.
  if (withSysMonitoring) {
    await createPackagePolicy(soClient, esClient, agentPolicyService, agentPolicyWithStagedSpaces, _common.FLEET_SYSTEM_PACKAGE, {
      spaceId,
      user,
      authorizationHeader,
      force: force || forcePackagePolicyCreation
    });
  }
  await (0, _api_keys.ensureDefaultEnrollmentAPIKeyForAgentPolicy)(soClient, esClient, agentPolicy.id);
  try {
    // Deploy policy will create the agentless agent if needed
    await agentPolicyService.deployPolicy(soClient, agentPolicy.id, undefined, {
      throwOnAgentlessError: true
    });
  } catch (err) {
    if (agentPolicy.supports_agentless) {
      await agentPolicyService.delete(soClient, esClient, agentPolicy.id).catch(() => {
        _2.appContextService.getLogger().error(`Error deleting agentless policy`, {
          error: agentPolicy
        });
      });
    }
    throw err;
  }
  return agentPolicy;
}