"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseExperimentalConfigValue = exports.isValidExperimentalValue = exports.getUpdatedExperimentalFeatures = exports.getExperimentalAllowedValues = exports.allowedExperimentalValues = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const _allowedExperimentalValues = {
  showExperimentalShipperOptions: false,
  useSpaceAwareness: true,
  enableAutomaticAgentUpgrades: true,
  enableSyncIntegrationsOnRemote: true,
  enableSSLSecrets: false,
  installedIntegrationsTabularUI: true,
  enabledUpgradeAgentlessDeploymentsTask: true,
  enablePackageRollback: true,
  enableAutoInstallContentPackages: true,
  enableOtelIntegrations: true,
  enableAgentStatusAlerting: true,
  enableAgentPrivilegeLevelChange: false,
  installIntegrationsKnowledge: false,
  agentlessPoliciesAPI: true,
  useAgentlessAPIInUI: false,
  disabledAgentlessLegacyAPI: false
};

/**
 * A list of allowed values that can be used in `xpack.fleet.enableExperimental` and `xpack.fleet.experimentalFeatures`.
 * This object is then used to validate and parse the value entered.
 */
const allowedExperimentalValues = exports.allowedExperimentalValues = Object.freeze({
  ..._allowedExperimentalValues
});
const allowedKeys = Object.keys(allowedExperimentalValues);

/**
 * Parses two settings in kibana configuration:
 * 1. xpack.fleet.enableExperimental: an array of experimental values to enable
 * 2. xpack.fleet.experimentalFeatures: an object that associates a boolean to experimental values to enable or disable them
 *
 * The objective is to make xpack.fleet.experimentalFeatures the source of truth, while keeping
 * xpack.fleet.enableExperimental for backward compatibility.
 * In case of conflict, xpack.fleet.experimentalFeatures takes precedence over xpack.fleet.enableExperimental.
 *
 * @param enableExperimentalConfigValue the value of xpack.fleet.enableExperimental
 * @param experimentalFeaturesConfigValue the value of xpack.fleet.experimentalFeatures
 * @returns an object with experimental features and their boolean values
 */
const parseExperimentalConfigValue = (enableExperimentalConfigValue, experimentalFeaturesConfigValue) => {
  return getUpdatedExperimentalFeatures(enableExperimentalConfigValue, experimentalFeaturesConfigValue, allowedExperimentalValues);
};

/**
 * Parses and merges two config values into a single object representing experimental features.
 * Made to be generic for easier testing.
 *
 * @param enableExperimentalConfigValue an array of strings that are experimental feature values
 * @param experimentalFeaturesConfigValue an object with string keys (experimental feature names) and boolean values
 * @param existingFeatures an object with pre-existing experimental features and their default values
 * @returns an object with experimental features values updated based on the two config values provided
 */
exports.parseExperimentalConfigValue = parseExperimentalConfigValue;
const getUpdatedExperimentalFeatures = (enableExperimentalConfigValue, experimentalFeaturesConfigValue, existingFeatures) => {
  const updatedFeatures = {
    ...existingFeatures
  };
  for (const value of enableExperimentalConfigValue) {
    if (Object.prototype.hasOwnProperty.call(existingFeatures, value)) {
      updatedFeatures[value] = true;
    }
  }
  for (const [key, value] of Object.entries(experimentalFeaturesConfigValue)) {
    if (Object.prototype.hasOwnProperty.call(existingFeatures, key) && typeof value === 'boolean') {
      updatedFeatures[key] = value;
    }
  }
  return updatedFeatures;
};
exports.getUpdatedExperimentalFeatures = getUpdatedExperimentalFeatures;
const isValidExperimentalValue = value => {
  return allowedKeys.includes(value);
};
exports.isValidExperimentalValue = isValidExperimentalValue;
const getExperimentalAllowedValues = () => [...allowedKeys];
exports.getExperimentalAllowedValues = getExperimentalAllowedValues;