"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EsIndexFilesMetadataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _fp = require("lodash/fp");
var _esQuery = require("@kbn/es-query");
var _pLimit = _interopRequireDefault(require("p-limit"));
var _utils = require("../../utils");
var _query_filters = require("./query_filters");
var _file = require("../../../saved_objects/file");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const filterArgsToESQuery = (0, _fp.pipe)(_query_filters.filterArgsToKuery, _esQuery.toElasticsearchQuery);
const bulkGetConcurrency = (0, _pLimit.default)(10);
const fileMappings = {
  dynamic: false,
  type: 'object',
  properties: {
    ..._file.fileObjectType.mappings.properties
  }
};
class EsIndexFilesMetadataClient {
  constructor(index, esClient, logger, indexIsAlias = false) {
    (0, _defineProperty2.default)(this, "createIfNotExists", (0, _lodash.once)(async () => {
      // We don't attempt to create the index if it is an Alias/DS
      if (this.indexIsAlias) {
        this.logger.debug(`No need to create index [${this.index}] as it is an Alias or DS.`);
        return;
      }
      try {
        if (await this.esClient.indices.exists({
          index: this.index
        })) {
          return;
        }
        await this.esClient.indices.create({
          index: this.index,
          mappings: {
            dynamic: false,
            properties: {
              file: fileMappings
            }
          }
        }).catch(_utils.wrapErrorAndReThrow.withMessagePrefix('EsIndexFilesMetadataClient.createIfNotExists(): '));
        this.logger.info(`index [${this.index}] created with default mappings.`);
      } catch (e) {
        this.logger.error(`Failed to create index [${this.index}]: ${e.message}`);
        this.logger.debug(e);
        // best effort
      }
    }));
    (0, _defineProperty2.default)(this, "attrPrefix", 'file');
    this.index = index;
    this.esClient = esClient;
    this.logger = logger;
    this.indexIsAlias = indexIsAlias;
  }
  async getBackingIndex(id) {
    var _doc$hits$hits, _doc$hits$hits$;
    if (!this.indexIsAlias) {
      return this.index;
    }
    const doc = await this.esClient.search({
      index: this.index,
      size: 1,
      query: {
        term: {
          _id: id
        }
      },
      _source: false // suppress the document content
    }).catch(_utils.wrapErrorAndReThrow.withMessagePrefix('EsIndexFilesMetadataClient.getBackingIndex(): '));
    const docIndex = (_doc$hits$hits = doc.hits.hits) === null || _doc$hits$hits === void 0 ? void 0 : (_doc$hits$hits$ = _doc$hits$hits[0]) === null || _doc$hits$hits$ === void 0 ? void 0 : _doc$hits$hits$._index;
    if (!docIndex) {
      const err = new Error(`Unable to determine backing index for file id [${id}] in index (alias) [${this.index}]`);
      this.logger.error(err);
      throw err;
    }
    return docIndex;
  }
  async create({
    id,
    metadata
  }) {
    await this.createIfNotExists();
    const result = await this.esClient.index({
      index: this.index,
      id,
      document: {
        file: metadata,
        // Add `@timestamp` if index is an Alias/DS
        ...(this.indexIsAlias ? {
          '@timestamp': new Date().toISOString()
        } : {})
      },
      op_type: 'create',
      refresh: 'wait_for'
    }).catch(_utils.wrapErrorAndReThrow.withMessagePrefix('EsIndexFilesMetadataClient.create(): '));
    return {
      id: result._id,
      metadata
    };
  }
  async get({
    id
  }) {
    const {
      esClient,
      index,
      indexIsAlias
    } = this;
    let doc;
    if (indexIsAlias) {
      var _await$esClient$searc, _await$esClient$searc2;
      doc = (_await$esClient$searc = (await esClient.search({
        index,
        size: 1,
        query: {
          term: {
            _id: id
          }
        }
      }).catch(_utils.wrapErrorAndReThrow.withMessagePrefix('EsIndexFilesMetadataClient.get(): '))).hits.hits) === null || _await$esClient$searc === void 0 ? void 0 : (_await$esClient$searc2 = _await$esClient$searc[0]) === null || _await$esClient$searc2 === void 0 ? void 0 : _await$esClient$searc2._source;
    } else {
      doc = (await esClient.get({
        index,
        id
      }).catch(_utils.wrapErrorAndReThrow.withMessagePrefix('EsIndexFilesMetadataClient.get(): ')))._source;
    }
    if (!doc) {
      this.logger.error(`File with id "${id}" not found in index ${indexIsAlias ? 'alias ' : ''}"${index}"`);
      throw new Error('File not found');
    }
    return {
      id,
      metadata: doc.file
    };
  }
  async bulkGet({
    ids,
    throwIfNotFound
  }) {
    const promises = ids.map(id => bulkGetConcurrency(() => this.get({
      id
    }).catch(e => {
      if (throwIfNotFound) {
        throw e;
      }
      return null;
    })));
    const result = await Promise.all(promises);
    return result;
  }
  async delete({
    id
  }) {
    await this.esClient.delete({
      index: this.index,
      id
    }).catch(_utils.wrapErrorAndReThrow.withMessagePrefix('EsIndexFilesMetadataClient.delete(): '));
  }
  async update({
    id,
    metadata
  }) {
    const index = await this.getBackingIndex(id);
    await this.esClient.update({
      index,
      id,
      doc: {
        file: metadata
      },
      refresh: 'wait_for'
    }).catch(_utils.wrapErrorAndReThrow.withMessagePrefix('EsIndexFilesMetadataClient.update(): '));
    return this.get({
      id
    });
  }
  paginationToES({
    page = 1,
    perPage = 50
  }) {
    return {
      size: perPage,
      from: (page - 1) * perPage
    };
  }
  async find({
    page,
    perPage,
    ...filterArgs
  } = {}) {
    const result = await this.esClient.search({
      track_total_hits: true,
      index: this.index,
      expand_wildcards: 'hidden',
      query: filterArgsToESQuery({
        ...filterArgs,
        attrPrefix: this.attrPrefix
      }),
      ...this.paginationToES({
        page,
        perPage
      }),
      sort: 'file.created'
    }).catch(_utils.wrapErrorAndReThrow.withMessagePrefix('EsIndexFilesMetadataClient.find(): '));
    return {
      total: result.hits.total.value,
      files: result.hits.hits.map(r => {
        var _r$_source;
        return {
          id: r._id,
          metadata: (_r$_source = r._source) === null || _r$_source === void 0 ? void 0 : _r$_source.file
        };
      })
    };
  }
  async getUsageMetrics(arg) {
    throw new Error('Not implemented');
  }
}
exports.EsIndexFilesMetadataClient = EsIndexFilesMetadataClient;