"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggestForEmptyExpression = suggestForEmptyExpression;
var _esqlTypes = require("@kbn/esql-types");
var _lodash = require("lodash");
var _utils = require("../utils");
var _comma_decision_engine = require("../comma_decision_engine");
var _functions = require("../../functions");
var _suggestion_builder = require("../suggestion_builder");
var _signature_analyzer = require("../signature_analyzer");
var _helpers = require("../../helpers");
var _values = require("../../../values");
var _constants = require("../../../../constants");
var _ = require("../../../../../..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** Handles suggestions when starting a new expression (empty position) */
async function suggestForEmptyExpression(ctx) {
  const {
    options
  } = ctx;
  const functionParamContext = options.functionParameterContext;
  if (functionParamContext) {
    return handleFunctionParameterContext(functionParamContext, ctx);
  }
  return handleDefaultContext(ctx);
}

/** Handles suggestions when cursor is inside a function parameter (empty position) */
async function handleFunctionParameterContext(functionParamContext, ctx) {
  // Early validation
  if (!isValidFunctionParameterPosition(functionParamContext)) {
    return [];
  }

  // Try exclusive suggestions first (COUNT(*), enum values)
  const exclusiveSuggestions = tryExclusiveSuggestions(functionParamContext, ctx);
  if (exclusiveSuggestions.length > 0) {
    return exclusiveSuggestions;
  }

  // Build composite suggestions (literals + fields + functions)
  return buildCompositeSuggestions(functionParamContext, ctx);
}

/** Validates that we can suggest for this function parameter position */
function isValidFunctionParameterPosition(functionParamContext) {
  var _functionDefinition$s;
  const {
    functionDefinition,
    paramDefinitions
  } = functionParamContext;
  if (!functionDefinition) {
    return false;
  }

  // Empty paramDefinitions = no valid signature for this position → return false
  // Example: FUNC(double, double, /) with only 1-2 param signatures → suggest nothing
  //
  // Exception (continue suggesting) only for variadic functions (CONCAT, COALESCE):
  // minParams defined, accepts unlimited params
  const isVariadicFunction = (_functionDefinition$s = functionDefinition.signatures) === null || _functionDefinition$s === void 0 ? void 0 : _functionDefinition$s.some(sig => sig.minParams != null);
  if (paramDefinitions.length === 0 && !isVariadicFunction) {
    return false;
  }
  return true;
}

/** Try suggestions that are exclusive (if present, return only these) */
function tryExclusiveSuggestions(functionParamContext, ctx) {
  var _options$isCursorFoll;
  const {
    functionDefinition,
    paramDefinitions
  } = functionParamContext;
  const {
    options
  } = ctx;
  const suggestions = [];

  // Special case: COUNT function suggests "*" (e.g., "COUNT(*)")
  if ((0, _utils.matchesSpecialFunction)(functionDefinition.name, 'count')) {
    suggestions.push(_.allStarConstant);
  }

  // Enum values are exclusive - if present, return only those
  const enumItems = buildEnumValueSuggestions(paramDefinitions, functionDefinition, Boolean(functionParamContext.hasMoreMandatoryArgs), (_options$isCursorFoll = options.isCursorFollowedByComma) !== null && _options$isCursorFoll !== void 0 ? _options$isCursorFoll : false);
  if (enumItems.length > 0) {
    return enumItems;
  }
  return suggestions;
}

/** Build composite suggestions: literals + fields + functions */
async function buildCompositeSuggestions(functionParamContext, ctx) {
  var _options$isCursorFoll2;
  const {
    options
  } = ctx;

  // Determine configuration
  const config = getParamSuggestionConfig(functionParamContext, (_options$isCursorFoll2 = options.isCursorFollowedByComma) !== null && _options$isCursorFoll2 !== void 0 ? _options$isCursorFoll2 : false);
  const suggestions = [];

  // Add literal suggestions
  suggestions.push(...buildLiteralSuggestions(functionParamContext, ctx, config));

  // Add field and function suggestions
  if (config.allowFieldsAndFunctions) {
    suggestions.push(...(await buildFieldAndFunctionSuggestions(functionParamContext, ctx, config)));
  }
  return suggestions;
}

/** Build all literal suggestions (constants + dates) */
function buildLiteralSuggestions(functionParamContext, ctx, config) {
  var _functionParamContext;
  const {
    paramDefinitions,
    functionDefinition
  } = functionParamContext;
  const {
    command
  } = ctx;
  const hasMoreMandatoryArgs = Boolean(functionParamContext.hasMoreMandatoryArgs);
  const suggestions = [];
  const hasConstantOnlyParams = paramDefinitions.some(({
    constantOnly
  }) => constantOnly);

  // Constant-only literals (true, false, null, string/number literals)
  const constantOnlySuggestions = buildConstantOnlyLiteralSuggestions(paramDefinitions, ctx, config.shouldAddComma, hasMoreMandatoryArgs);
  suggestions.push(...constantOnlySuggestions);

  // Date literals (now(), 1 hour, 2 days, ?_tstart, ?_tend) - only add if not already added by constantOnly path
  // Skip for:
  // - FTS functions
  // - BUCKET first parameter (field) in STATS
  // - When constantOnly params exist (already added via getCompatibleLiterals)
  const isFtsFunction = _constants.FULL_TEXT_SEARCH_FUNCTIONS.includes(functionDefinition.name);
  const isBucketFirstParam = (0, _utils.matchesSpecialFunction)(functionDefinition.name, 'bucket') && command.name === 'stats' && ((_functionParamContext = functionParamContext.currentParameterIndex) !== null && _functionParamContext !== void 0 ? _functionParamContext : 0) === 0;
  if (!isFtsFunction && !isBucketFirstParam && !hasConstantOnlyParams) {
    const builder = new _suggestion_builder.SuggestionBuilder(ctx);
    builder.addLiterals({
      types: paramDefinitions.map(({
        type
      }) => type),
      includeDateLiterals: true,
      includeCompatibleLiterals: false,
      addComma: config.shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
    });
    suggestions.push(...builder.build());
  }
  return suggestions;
}

/** Build field and function suggestions using SuggestionBuilder */
async function buildFieldAndFunctionSuggestions(functionParamContext, ctx, config) {
  const {
    paramDefinitions
  } = functionParamContext;
  const {
    options
  } = ctx;
  const ignoredColumns = options.ignoredColumnsForEmptyExpression || [];
  const builder = new _suggestion_builder.SuggestionBuilder(ctx);

  // Suggest fields when:
  // - there is at least one non-constant parameter, OR
  // - param definitions are empty (variadic/unknown position, e.g., CONCAT third+ arg)

  const hasConstantOnlyParam = paramDefinitions.some(({
    constantOnly
  }) => constantOnly);
  const hasFieldsOnlyParam = paramDefinitions.some(({
    fieldsOnly
  }) => fieldsOnly);

  // constantOnly params require literal values, not fields
  // (variadic functions work correctly: empty paramDefinitions → hasConstantOnlyParam = false)
  if (!hasConstantOnlyParam) {
    const canBeMultiValue = paramDefinitions.some(t => t && (t.supportsMultiValues === true || t.name === 'values'));
    await builder.addFields({
      types: config.acceptedTypes,
      ignoredColumns,
      addComma: config.shouldAddComma,
      promoteToTop: true,
      canBeMultiValue
    });
  }

  // constantOnly params require literal values, not function results
  if (!hasFieldsOnlyParam && !hasConstantOnlyParam) {
    builder.addFunctions({
      types: config.acceptedTypes,
      ignoredFunctions: functionParamContext.functionsToIgnore || [],
      addComma: config.shouldAddComma
    });
  }
  return builder.build();
}

/** Handles suggestions for top-level expression (not inside a function parameter) */
async function handleDefaultContext(ctx) {
  var _options$addSpaceAfte, _options$suggestField, _options$suggestFunct, _options$controlType;
  const {
    context,
    options
  } = ctx;
  const ignoredColumns = options.ignoredColumnsForEmptyExpression || [];
  const addSpaceAfterField = (_options$addSpaceAfte = options.addSpaceAfterFirstField) !== null && _options$addSpaceAfte !== void 0 ? _options$addSpaceAfte : true;
  const suggestFields = (_options$suggestField = options.suggestFields) !== null && _options$suggestField !== void 0 ? _options$suggestField : true;
  const suggestFunctions = (_options$suggestFunct = options.suggestFunctions) !== null && _options$suggestFunct !== void 0 ? _options$suggestFunct : true;
  const controlType = (_options$controlType = options.controlType) !== null && _options$controlType !== void 0 ? _options$controlType : _esqlTypes.ESQLVariableType.FIELDS;
  const suggestions = [];
  const acceptedTypes = ['any'];

  // Suggest fields/columns and functions using SuggestionBuilder
  if (suggestFields || suggestFunctions) {
    const builder = new _suggestion_builder.SuggestionBuilder(ctx);
    if (suggestFields) {
      await builder.addFields({
        types: acceptedTypes,
        ignoredColumns,
        addSpaceAfterField,
        promoteToTop: true,
        ...(options.openSuggestions !== undefined && {
          openSuggestions: options.openSuggestions
        })
      });
    }
    if (suggestFunctions) {
      builder.addFunctions({
        types: acceptedTypes,
        ignoredFunctions: [],
        ...(options.openSuggestions !== undefined && {
          openSuggestions: options.openSuggestions
        })
      });
    }
    suggestions.push(...builder.build());
  }

  // Suggest control variables (e.g., "?fieldName", "$valueName") if supported and not already present
  if (context !== null && context !== void 0 && context.supportsControls) {
    const hasControl = suggestions.some(suggestion => {
      var _suggestion$command, _suggestion$command$i;
      return (_suggestion$command = suggestion.command) === null || _suggestion$command === void 0 ? void 0 : (_suggestion$command$i = _suggestion$command.id) === null || _suggestion$command$i === void 0 ? void 0 : _suggestion$command$i.includes('esql.control');
    });
    if (!hasControl) {
      var _context$variables$fi, _context$variables;
      const prefix = (0, _helpers.getVariablePrefix)(controlType);
      const variableNames = (_context$variables$fi = (_context$variables = context.variables) === null || _context$variables === void 0 ? void 0 : _context$variables.filter(({
        type
      }) => type === controlType).map(({
        key
      }) => `${prefix}${key}`)) !== null && _context$variables$fi !== void 0 ? _context$variables$fi : [];
      const controlSuggestions = (0, _helpers.getControlSuggestion)(controlType, _esqlTypes.ControlTriggerSource.SMART_SUGGESTION, variableNames);
      suggestions.push(...controlSuggestions);
    }
  }
  return suggestions;
}

/** Collects all unique suggested values from parameter definitions (e.g., enums) */
function collectSuggestedValues(paramDefinitions) {
  return (0, _lodash.uniq)(paramDefinitions.map(({
    suggestedValues
  }) => suggestedValues).filter(values => Boolean(values)).flat());
}

/** Filters parameters that only accept constant values (literals or duration types) */
function getConstantOnlyParams(paramDefinitions) {
  return paramDefinitions.filter(({
    constantOnly,
    type
  }) => constantOnly || /_duration/.test(String(type)));
}

/** Derives suggestion configuration for next function parameter */
function getParamSuggestionConfig(functionParamContext, isCursorFollowedByComma) {
  var _analyzer$getAccepted;
  const {
    functionDefinition
  } = functionParamContext;
  const analyzer = _signature_analyzer.SignatureAnalyzer.from(functionParamContext);
  const acceptedTypes = (_analyzer$getAccepted = analyzer === null || analyzer === void 0 ? void 0 : analyzer.getAcceptedTypes()) !== null && _analyzer$getAccepted !== void 0 ? _analyzer$getAccepted : ['any'];
  const hasMoreMandatoryArgs = Boolean(functionParamContext.hasMoreMandatoryArgs);
  const commaContext = {
    position: 'empty_expression',
    hasMoreMandatoryArgs,
    functionType: functionDefinition.type,
    isCursorFollowedByComma,
    functionSignatures: functionDefinition.signatures
  };
  const shouldAddComma = (0, _comma_decision_engine.shouldSuggestComma)(commaContext);
  const allowFieldsAndFunctions = true;
  return {
    acceptedTypes,
    shouldAddComma,
    allowFieldsAndFunctions
  };
}

/** Builds suggestions for enum/predefined values */
function buildEnumValueSuggestions(paramDefinitions, functionDefinition, hasMoreMandatoryArgs, isCursorFollowedByComma) {
  const values = collectSuggestedValues(paramDefinitions);
  if (!values.length) {
    return [];
  }
  const commaContext = {
    position: 'enum_value',
    hasMoreMandatoryArgs,
    functionType: functionDefinition.type,
    isCursorFollowedByComma,
    functionSignatures: functionDefinition.signatures
  };
  const shouldAddComma = (0, _comma_decision_engine.shouldSuggestComma)(commaContext);
  return (0, _values.buildValueDefinitions)(values, {
    addComma: shouldAddComma,
    advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
  });
}

/** Builds suggestions for constant-only literal parameters */
function buildConstantOnlyLiteralSuggestions(paramDefinitions, ctx, shouldAddComma, hasMoreMandatoryArgs) {
  const constantOnlyParams = getConstantOnlyParams(paramDefinitions);
  if (!constantOnlyParams.length) {
    return [];
  }
  const types = (0, _functions.ensureKeywordAndText)(constantOnlyParams.map(({
    type
  }) => type));
  const builder = new _suggestion_builder.SuggestionBuilder(ctx);
  builder.addLiterals({
    types,
    addComma: shouldAddComma,
    advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs,
    includeDateLiterals: false,
    // Date literals are added separately in buildLiteralSuggestions
    includeCompatibleLiterals: true
  });
  builder.addFunctions({
    types,
    addComma: shouldAddComma,
    constantGeneratingOnly: true
  });
  return builder.build();
}