"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autocomplete = autocomplete;
var _i18n = require("@kbn/i18n");
var _helpers = require("../../../definitions/utils/autocomplete/helpers");
var _types = require("../../types");
var _complete_items = require("../../complete_items");
var _utils = require("./utils");
var _sources = require("../../../definitions/utils/sources");
var _ = require("../..");
var _utils2 = require("../../../definitions/utils");
var _is = require("../../../ast/is");
var _location = require("../../../ast/location");
var _expressions = require("../../../definitions/utils/expressions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function autocomplete(query, command, callbacks, context, cursorPosition = query.length) {
  if (!(callbacks !== null && callbacks !== void 0 && callbacks.getByType) || !(callbacks !== null && callbacks !== void 0 && callbacks.getColumnsForQuery)) {
    return [];
  }
  const innerText = query.substring(0, cursorPosition);
  let commandText = innerText;
  if (command.location) {
    commandText = innerText.slice(command.location.min);
  }
  const position = (0, _utils.getPosition)(commandText, command, cursorPosition);
  switch (position.pos) {
    case 'type':
    case 'after_type':
    case 'mnemonic':
      {
        const joinCommandDefinition = _.esqlCommandRegistry.getCommandByName('join');
        const allMnemonics = (0, _utils.getFullCommandMnemonics)(joinCommandDefinition);
        const filteredMnemonics = allMnemonics.filter(([mnemonic]) => mnemonic.startsWith(commandText.toUpperCase()));
        if (!filteredMnemonics.length) {
          return [];
        }
        return filteredMnemonics.map(([mnemonic, description], i) => (0, _helpers.withAutoSuggest)({
          label: mnemonic,
          text: mnemonic + ' $0',
          asSnippet: true,
          detail: description,
          kind: 'Keyword',
          sortText: `${i}-MNEMONIC`
        }));
      }
    case 'after_mnemonic':
    case 'index':
      {
        var _commandText$split$po, _await$callbacks$canC, _callbacks$canCreateL;
        const indexNameInput = (_commandText$split$po = commandText.split(' ').pop()) !== null && _commandText$split$po !== void 0 ? _commandText$split$po : '';
        const joinSources = context === null || context === void 0 ? void 0 : context.joinSources;
        const suggestions = [];
        const canCreate = (_await$callbacks$canC = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$canCreateL = callbacks.canCreateLookupIndex) === null || _callbacks$canCreateL === void 0 ? void 0 : _callbacks$canCreateL.call(callbacks, indexNameInput))) !== null && _await$callbacks$canC !== void 0 ? _await$callbacks$canC : false;
        const indexAlreadyExists = joinSources === null || joinSources === void 0 ? void 0 : joinSources.some(source => source.name === indexNameInput || source.aliases.includes(indexNameInput));
        if (canCreate && !indexAlreadyExists) {
          const createIndexCommandSuggestion = (0, _helpers.getLookupIndexCreateSuggestion)(innerText, indexNameInput);
          suggestions.push(createIndexCommandSuggestion);
        }
        if (joinSources !== null && joinSources !== void 0 && joinSources.length) {
          const joinIndexesSuggestions = (0, _sources.specialIndicesToSuggestions)(joinSources);
          suggestions.push(...(await (0, _helpers.handleFragment)(innerText, fragment => (0, _sources.specialIndicesToSuggestions)(joinSources).some(({
            label
          }) => label.toLocaleLowerCase() === fragment.toLocaleLowerCase()), (_fragment, rangeToReplace) => joinIndexesSuggestions.map(suggestion => ({
            ...suggestion,
            rangeToReplace
          })), () => [])));
        }
        return suggestions;
      }
    case 'after_index':
      {
        const suggestion = (0, _helpers.withAutoSuggest)({
          label: 'ON',
          text: 'ON ',
          detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.join.onKeyword', {
            defaultMessage: 'Specify JOIN field conditions'
          }),
          kind: 'Keyword',
          sortText: '0-ON'
        });
        return [suggestion];
      }
    case 'on_expression':
      {
        var _callbacks$getByType, _callbacks$getColumns, _callbacks$getColumns2;
        const joinCommand = command;
        const expressionRoot = position.expression;

        // Create enriched getByType that includes lookup fields
        const enrichedGetByType = await (0, _utils.createEnrichedGetByType)((_callbacks$getByType = callbacks === null || callbacks === void 0 ? void 0 : callbacks.getByType) !== null && _callbacks$getByType !== void 0 ? _callbacks$getByType : () => Promise.resolve([]), joinCommand, (_callbacks$getColumns = callbacks === null || callbacks === void 0 ? void 0 : callbacks.getColumnsForQuery) !== null && _callbacks$getColumns !== void 0 ? _callbacks$getColumns : () => Promise.resolve([]), context);

        // Create enriched context that includes lookup fields in columns map
        const enrichedContext = await (0, _utils.createEnrichedContext)(context, joinCommand, (_callbacks$getColumns2 = callbacks === null || callbacks === void 0 ? void 0 : callbacks.getColumnsForQuery) !== null && _callbacks$getColumns2 !== void 0 ? _callbacks$getColumns2 : () => Promise.resolve([]));
        const suggestions = await (0, _utils2.suggestForExpression)({
          query,
          expressionRoot,
          command,
          cursorPosition,
          location: _types.Location.JOIN,
          context: enrichedContext,
          callbacks: {
            ...callbacks,
            getByType: enrichedGetByType
          },
          options: {
            preferredExpressionType: 'boolean'
          }
        });

        // Filter out AS operator - it's not valid in boolean expressions
        const filteredSuggestions = suggestions.filter(({
          label
        }) => label !== 'AS');
        const insideFunction = expressionRoot && (0, _is.isFunctionExpression)(expressionRoot) && (0, _location.within)(cursorPosition, expressionRoot);
        if (expressionRoot && !insideFunction) {
          const expressionType = (0, _expressions.getExpressionType)(expressionRoot, enrichedContext === null || enrichedContext === void 0 ? void 0 : enrichedContext.columns);
          const isBooleanComplete = expressionType === 'boolean' && (0, _expressions.isExpressionComplete)(expressionType, innerText);

          // Special case: single common field (exists in both source and lookup) is valid as shorthand for field = field
          const fieldIsCommon = expressionRoot.type === 'column' && (0, _utils.isCommonField)(expressionRoot.name, context);
          if (isBooleanComplete || !isBooleanComplete && fieldIsCommon) {
            filteredSuggestions.push((0, _helpers.withAutoSuggest)({
              ..._complete_items.commaCompleteItem,
              text: ', '
            }));
            filteredSuggestions.push(_complete_items.pipeCompleteItem);
          }
        }
        return filteredSuggestions;
      }
  }
  const suggestions = [];
  return suggestions;
}