"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DEFAULT_PROJECT_ROUTING = exports.CPSManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _cpsUtils = require("@kbn/cps-utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
/**
 * This should be configured on spaces level.
 * Common values: PROJECT_ROUTING.ALL (all projects, will be parsed to undefined on request level), '_alias:_origin' (origin project only)
 */
const DEFAULT_PROJECT_ROUTING = exports.DEFAULT_PROJECT_ROUTING = _cpsUtils.PROJECT_ROUTING.ALL;

/**
 * Central service for managing project routing and project data.
 *
 * - Fetches project data from ES via `/internal/cps/projects_tags` endpoint (with caching and retry logic)
 * - Manages current project routing state using observables
 * - projectRouting$ represents temporary UI state; apps should reset to their saved value or DEFAULT_PROJECT_ROUTING on navigation
 */
class CPSManager {
  constructor(deps) {
    (0, _defineProperty2.default)(this, "http", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "application", void 0);
    (0, _defineProperty2.default)(this, "projectFetcherPromise", null);
    (0, _defineProperty2.default)(this, "projectRouting$", new _rxjs.BehaviorSubject(DEFAULT_PROJECT_ROUTING));
    (0, _defineProperty2.default)(this, "projectPickerAccess$", void 0);
    this.http = deps.http;
    this.logger = deps.logger.get('cps_manager');
    this.application = deps.application;
    this.projectPickerAccess$ = (0, _rxjs.combineLatest)([this.application.currentAppId$, this.application.currentLocation$]).pipe((0, _rxjs.switchMap)(async ([appId, location]) => {
      // Lazy load access control only when observable is subscribed
      const {
        getProjectRoutingAccess,
        getReadonlyMessage
      } = await Promise.resolve().then(() => _interopRequireWildcard(require('./async_services')));
      const access = getProjectRoutingAccess(appId !== null && appId !== void 0 ? appId : '', location !== null && location !== void 0 ? location : '');
      return {
        access,
        readonlyMessage: getReadonlyMessage(appId)
      };
    }));
  }

  /**
   * Get the current project routing as an observable
   */
  getProjectRouting$() {
    return this.projectRouting$.asObservable();
  }

  /**
   * Set the current project routing
   */
  setProjectRouting(projectRouting) {
    this.projectRouting$.next(projectRouting);
  }

  /**
   * Get the current project routing value
   */
  getProjectRouting() {
    return this.projectRouting$.value;
  }

  /**
   * Get the default project routing value.
   * This is the fallback value used when no app-specific or saved value exists.
   */
  getDefaultProjectRouting() {
    return DEFAULT_PROJECT_ROUTING;
  }

  /**
   * Get the project picker access level as an observable.
   * This combines the current app ID and location to determine whether
   * the project picker should be editable, readonly, or disabled.
   * Also returns the custom readonly message if applicable.
   */
  getProjectPickerAccess$() {
    return this.projectPickerAccess$;
  }

  /**
   * Fetches projects from the server with caching and retry logic.
   * Returns cached data if already loaded. If a fetch is already in progress, returns the existing promise.
   * @returns Promise resolving to ProjectsData
   */
  async fetchProjects() {
    const fetcher = await this.getProjectFetcher();
    return fetcher.fetchProjects();
  }

  /**
   * Forces a refresh of projects from the server, bypassing the cache.
   * @returns Promise resolving to ProjectsData
   */
  async refresh() {
    const fetcher = await this.getProjectFetcher();
    return fetcher.refresh();
  }
  async getProjectFetcher() {
    if (!this.projectFetcherPromise) {
      this.projectFetcherPromise = Promise.resolve().then(() => _interopRequireWildcard(require('./async_services'))).then(({
        createProjectFetcher
      }) => createProjectFetcher(this.http, this.logger));
    }
    return this.projectFetcherPromise;
  }
}
exports.CPSManager = CPSManager;