"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createServerlessKibana = createServerlessKibana;
exports.createTestServerlessInstances = createTestServerlessInstances;
var _path = _interopRequireDefault(require("path"));
var _lodash = require("lodash");
var _elasticsearch = require("@elastic/elasticsearch");
var _es = require("@kbn/es");
var _repoInfo = require("@kbn/repo-info");
var _toolingLog = require("@kbn/tooling-log");
var _test = require("@kbn/test");
var _devUtils = require("@kbn/dev-utils");
var _create_root = require("./create_root");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const ES_BASE_PATH_DIR = _path.default.join(_repoInfo.REPO_ROOT, '.es/es_test_serverless');
const projectType = 'es';

/**
 * See docs in {@link TestUtils}. This function provides the same utilities but
 * configured for serverless.
 *
 * @note requires a Docker installation to be running
 */
function createTestServerlessInstances({
  adjustTimeout,
  kibana = {}
} = {}) {
  adjustTimeout === null || adjustTimeout === void 0 ? void 0 : adjustTimeout(150_000);
  const esUtils = createServerlessES();
  const kbUtils = createServerlessKibana(kibana.settings, kibana.cliArgs);
  return {
    startES: async () => {
      const {
        stop,
        getClient
      } = await esUtils.start();
      return {
        es: esUtils.es,
        getClient,
        stop
      };
    },
    startKibana: async abortSignal => {
      abortSignal === null || abortSignal === void 0 ? void 0 : abortSignal.addEventListener('abort', async () => await kbUtils.shutdown());
      await kbUtils.preboot();
      const coreSetup = await kbUtils.setup();
      const coreStart = await kbUtils.start();
      return {
        root: kbUtils,
        coreSetup,
        coreStart,
        stop: kbUtils.shutdown.bind(kbUtils)
      };
    }
  };
}
function createServerlessES() {
  const log = new _toolingLog.ToolingLog({
    level: 'info',
    writeTo: process.stdout
  });
  const es = new _es.Cluster({
    log
  });
  const esPort = _test.esTestConfig.getPort();
  const esServerlessImageParams = parseEsServerlessImageOverride(_test.esTestConfig.getESServerlessImage());
  return {
    es,
    start: async () => {
      await es.runServerless({
        projectType,
        basePath: ES_BASE_PATH_DIR,
        port: esPort,
        background: true,
        clean: true,
        kill: true,
        waitForReady: true,
        ...esServerlessImageParams
      });
      const client = getServerlessESClient({
        port: esPort
      });
      return {
        getClient: () => client,
        stop: async () => {
          await es.stop();
        }
      };
    }
  };
}
const getServerlessESClient = ({
  port
}) => {
  return new _elasticsearch.Client({
    node: `http://localhost:${port}`,
    Connection: _elasticsearch.HttpConnection,
    requestTimeout: 30_000,
    auth: {
      ..._test.systemIndicesSuperuser
    }
  });
};
const getServerlessDefault = () => {
  return {
    server: {
      restrictInternalApis: true,
      // has no effect, defaults to true
      versioned: {
        versionResolution: 'newest',
        strictClientVersionCheck: false
      }
    },
    elasticsearch: {
      hosts: [`http://localhost:${_test.esTestConfig.getPort()}`],
      serviceAccountToken: _devUtils.kibanaDevServiceAccount.token
    },
    migrations: {
      algorithm: 'zdt',
      zdt: {
        runOnRoles: ['ui']
      }
    },
    logging: {
      loggers: [{
        name: 'root',
        level: 'error',
        appenders: ['console']
      }, {
        name: 'elasticsearch.deprecation',
        level: 'all',
        appenders: ['deprecation']
      }],
      appenders: {
        deprecation: {
          type: 'console',
          layout: {
            type: 'json'
          }
        },
        console: {
          type: 'console',
          layout: {
            type: 'pattern'
          }
        }
      }
    }
  };
};
function createServerlessKibana(settings = {}, cliArgs = {}) {
  return (0, _create_root.createRoot)((0, _lodash.defaultsDeep)(settings, getServerlessDefault()), {
    ...cliArgs,
    serverless: true
  });
}
function parseEsServerlessImageOverride(dockerImageOrTag) {
  if (!dockerImageOrTag) {
    return {};
  } else if (dockerImageOrTag.includes(':')) {
    return {
      image: dockerImageOrTag
    };
  } else {
    return {
      tag: dockerImageOrTag
    };
  }
}