"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WidgetType = exports.UISchemas = void 0;
exports.getMeta = getMeta;
var _v = require("@kbn/zod/v4");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
/**
 * Stack Connectors 2.0 - UI Metadata Extension System
 *
 * This file contains the Zod metadata extension system that enables
 * UI to be fully derived from schemas.
 *
 * WHY SEPARATE FILE: UI concerns are orthogonal to connector logic.
 * This allows the main spec to focus on connector behavior while
 * UI derivation is handled here.
 */
// ============================================================================
// ZOD METADATA EXTENSIONS
// ============================================================================
let WidgetType = exports.WidgetType = /*#__PURE__*/function (WidgetType) {
  WidgetType["Text"] = "text";
  WidgetType["Password"] = "password";
  WidgetType["Select"] = "select";
  WidgetType["FormFieldset"] = "formFieldset";
  WidgetType["KeyValue"] = "keyValue";
  return WidgetType;
}({});
function getMeta(schema) {
  return _v.z.globalRegistry.get(schema);
}

/**
 * Extended Zod type definition with UI metadata
 *
 * WHY: Zod schemas alone can derive most UI (field type, validation, labels),
 * but some UI concerns can't be inferred from types alone:
 * - Is this string a password? (needs masking)
 * - Should this string be a textarea or single-line input?
 * - What's the placeholder text? (different from label)
 * - Which section does this field belong to?
 *
 * This metadata extension allows schemas to carry UI hints while remaining
 * completely optional - fields without metadata get sensible defaults.
 */

/**
 * Pre-configured schema types for common field patterns
 * WHY: Reduces boilerplate for frequently used field types
 * These use Zod's built-in .meta() to attach UI metadata
 *
 * @example
 * const apiKey = z.string().meta({ sensitive: true, placeholder: "sk-..." });
 * // Or use the helper:
 * const apiKey = UISchemas.secret("sk-...");
 */
const UISchemas = exports.UISchemas = {
  /**
   * Secret/password field - automatically masked in UI
   * USED BY: All connectors with API keys/tokens (Slack, OpenAI, Jira, etc.)
   * @example secrets: { apiKey: UISchemas.secret().describe("API Key") }
   */
  secret: placeholder => _v.z.string().meta({
    sensitive: true,
    widget: 'password',
    placeholder
  }),
  /**
   * Multi-line text field
   * USED BY: Webhook, Slack (message body), Teams (message text)
   * @example message: UISchemas.textarea({ rows: 5 }).describe("Message body")
   */
  textarea: options => _v.z.string().meta({
    widget: 'textarea',
    widgetOptions: options
  }),
  /**
   * JSON editor field with syntax highlighting
   * USED BY: Webhook (body), OpenAI (functions), Slack (blocks)
   * @example config: UISchemas.json().describe("Configuration object")
   */
  json: () => _v.z.string().meta({
    widget: 'json'
  }),
  /**
   * Code editor field
   * USED BY: Tines (custom scripts), TheHive (case templates)
   * @example script: UISchemas.code("javascript").describe("Custom script")
   */
  code: language => _v.z.string().meta({
    widget: 'code',
    widgetOptions: {
      language
    }
  }),
  /**
   * URL field with validation
   * USED BY: All webhook-based connectors (Webhook, Cases Webhook, Slack webhook)
   * @example webhookUrl: UISchemas.url().describe("Webhook URL")
   */
  url: placeholder => _v.z.url().meta({
    widget: 'text',
    placeholder: placeholder !== null && placeholder !== void 0 ? placeholder : 'https://'
  })
};