// src/server/error.ts
var A2AError = class _A2AError extends Error {
  code;
  data;
  taskId;
  // Optional task ID context
  constructor(code, message, data, taskId) {
    super(message);
    this.name = "A2AError";
    this.code = code;
    this.data = data;
    this.taskId = taskId;
  }
  /**
   * Formats the error into a standard JSON-RPC error object structure.
   */
  toJSONRPCError() {
    const errorObject = {
      code: this.code,
      message: this.message
    };
    if (this.data !== void 0) {
      errorObject.data = this.data;
    }
    return errorObject;
  }
  // Static factory methods for common errors
  static parseError(message, data) {
    return new _A2AError(-32700, message, data);
  }
  static invalidRequest(message, data) {
    return new _A2AError(-32600, message, data);
  }
  static methodNotFound(method) {
    return new _A2AError(
      -32601,
      `Method not found: ${method}`
    );
  }
  static invalidParams(message, data) {
    return new _A2AError(-32602, message, data);
  }
  static internalError(message, data) {
    return new _A2AError(-32603, message, data);
  }
  static taskNotFound(taskId) {
    return new _A2AError(
      -32001,
      `Task not found: ${taskId}`,
      void 0,
      taskId
    );
  }
  static taskNotCancelable(taskId) {
    return new _A2AError(
      -32002,
      `Task not cancelable: ${taskId}`,
      void 0,
      taskId
    );
  }
  static pushNotificationNotSupported() {
    return new _A2AError(
      -32003,
      "Push Notification is not supported"
    );
  }
  static unsupportedOperation(operation) {
    return new _A2AError(
      -32004,
      `Unsupported operation: ${operation}`
    );
  }
  static authenticatedExtendedCardNotConfigured() {
    return new _A2AError(
      -32007,
      `Extended card not configured.`
    );
  }
};

// src/server/transports/jsonrpc_transport_handler.ts
var JsonRpcTransportHandler = class {
  requestHandler;
  constructor(requestHandler) {
    this.requestHandler = requestHandler;
  }
  /**
   * Handles an incoming JSON-RPC request.
   * For streaming methods, it returns an AsyncGenerator of JSONRPCResult.
   * For non-streaming methods, it returns a Promise of a single JSONRPCMessage (Result or ErrorResponse).
   */
  async handle(requestBody) {
    let rpcRequest;
    try {
      if (typeof requestBody === "string") {
        rpcRequest = JSON.parse(requestBody);
      } else if (typeof requestBody === "object" && requestBody !== null) {
        rpcRequest = requestBody;
      } else {
        throw A2AError.parseError("Invalid request body type.");
      }
      if (rpcRequest.jsonrpc !== "2.0" || !rpcRequest.method || typeof rpcRequest.method !== "string") {
        throw A2AError.invalidRequest(
          "Invalid JSON-RPC request structure."
        );
      }
    } catch (error) {
      const a2aError = error instanceof A2AError ? error : A2AError.parseError(error.message || "Failed to parse JSON request.");
      return {
        jsonrpc: "2.0",
        id: typeof rpcRequest?.id !== "undefined" ? rpcRequest.id : null,
        error: a2aError.toJSONRPCError()
      };
    }
    const { method, id: requestId = null } = rpcRequest;
    try {
      if (method === "agent/getAuthenticatedExtendedCard") {
        const result = await this.requestHandler.getAuthenticatedExtendedAgentCard();
        return {
          jsonrpc: "2.0",
          id: requestId,
          result
        };
      }
      if (!rpcRequest.params) {
        throw A2AError.invalidParams(`'params' is required for '${method}'`);
      }
      if (method === "message/stream" || method === "tasks/resubscribe") {
        const params = rpcRequest.params;
        const agentCard = await this.requestHandler.getAgentCard();
        if (!agentCard.capabilities.streaming) {
          throw A2AError.unsupportedOperation(`Method ${method} requires streaming capability.`);
        }
        const agentEventStream = method === "message/stream" ? this.requestHandler.sendMessageStream(params) : this.requestHandler.resubscribe(params);
        return async function* jsonRpcEventStream() {
          try {
            for await (const event of agentEventStream) {
              yield {
                jsonrpc: "2.0",
                id: requestId,
                // Use the original request ID for all streamed responses
                result: event
              };
            }
          } catch (streamError) {
            console.error(`Error in agent event stream for ${method} (request ${requestId}):`, streamError);
            throw streamError;
          }
        }();
      } else {
        let result;
        switch (method) {
          case "message/send":
            result = await this.requestHandler.sendMessage(rpcRequest.params);
            break;
          case "tasks/get":
            result = await this.requestHandler.getTask(rpcRequest.params);
            break;
          case "tasks/cancel":
            result = await this.requestHandler.cancelTask(rpcRequest.params);
            break;
          case "tasks/pushNotificationConfig/set":
            result = await this.requestHandler.setTaskPushNotificationConfig(
              rpcRequest.params
            );
            break;
          case "tasks/pushNotificationConfig/get":
            result = await this.requestHandler.getTaskPushNotificationConfig(
              rpcRequest.params
            );
            break;
          case "tasks/pushNotificationConfig/delete":
            await this.requestHandler.deleteTaskPushNotificationConfig(
              rpcRequest.params
            );
            result = null;
            break;
          case "tasks/pushNotificationConfig/list":
            result = await this.requestHandler.listTaskPushNotificationConfigs(
              rpcRequest.params
            );
            break;
          default:
            throw A2AError.methodNotFound(method);
        }
        return {
          jsonrpc: "2.0",
          id: requestId,
          result
        };
      }
    } catch (error) {
      const a2aError = error instanceof A2AError ? error : A2AError.internalError(error.message || "An unexpected error occurred.");
      return {
        jsonrpc: "2.0",
        id: requestId,
        error: a2aError.toJSONRPCError()
      };
    }
  }
};

export {
  A2AError,
  JsonRpcTransportHandler
};
