"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.artifactService = exports.Artifact = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _crypto = require("crypto");
var _axios = _interopRequireDefault(require("axios"));
var _lodash = require("lodash");
var _admZip = _interopRequireDefault(require("adm-zip"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Interface for managing artifact fetching and retrieval.
 *
 * Implementations must define how to start the service with a given telemetry receiver,
 * fetch individual artifacts by name, and provide the manifest URL in use.
 */

/**
 * Describes the shape of an artifact manifest and its modification state.
 *
 * @property data - The actual manifest data, format depends on implementation.
 * @property notModified - Indicates whether the manifest data has changed since last retrieval.
 */

/**
 * An entry in the manifest cache that stores the manifest and its associated ETag value.
 *
 * @property manifest - The Manifest object representing the artifact manifest data and status.
 * @property etag - The ETag string returned by the CDN for cache validation.
 */

/**
 * Configuration details for the CDN used to fetch artifacts.
 *
 * @property url - The base URL of the CDN for artifact downloads.
 * @property pubKey - The public key string used to verify artifact signatures.
 */

const DEFAULT_CDN_CONFIG = {
  url: 'https://artifacts.security.elastic.co',
  pubKey: `
-----BEGIN PUBLIC KEY-----
MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA6AB2sJ5M1ImN/bkQ7Te6
uI7vMXjN2yupEmh2rYz4gNzWS351d4JOhuQH3nzxfKdayHgusP/Kq2MXVqALH8Ru
Yu2AF08GdvYlQXPgEVI+tB/riekwU7PXZHdA1dY5/mEZ8SUSM25kcDJ3vTCzFTlL
gl2RNAdkR80d9nhvNSWlhWMwr8coQkr6NmujVU/Wa0w0EXbN1arjcG4qzbOCaR+b
cgQ9LRUoFfK9w+JJHDNjOI7rOmaIDA6Ep4oeDLy5AcGCE8bNmQzxZhRW7NvlNUGS
NTgU0CZTatVsL9AyP15W3k635Cpmy2SMPX+d/CFgvr8QPxtqdrz3q9iOeU3a1LMY
gDcFVmSzn5zieQEPfo/FcQID/gnCmkX0ADVMf1Q20ew66H7UCOejGaerbFZXYnTz
5AgQBWF2taOSSE7gDjGAHereeKp+1PR+tCkoDZIrPEjo0V6+KaTMuYS3oZj1/RZN
oTjQrdfeDj02mEIL+XkcWKAp03PYlWylVwgTMa178DDVuTWtS5lZL8j5LijlH9+6
xH8o++ghwfxp6ENLKDZPV5IvHHG7Vth9HScoPTQWQ+s8Bt26QENPUV2AbyxbJykY
mJfTDke3bEemHZzRbAmwiQ7VpJjJ4OfLGRy8Pp2AHo8kYIvWyM5+aLMxcxUaYdA9
5SxoDOgcDBA4lLb6XFLYiDUCAwEAAQ==
-----END PUBLIC KEY-----
`
};
class Artifact {
  constructor() {
    (0, _defineProperty2.default)(this, "manifestUrl", void 0);
    (0, _defineProperty2.default)(this, "cdn", void 0);
    (0, _defineProperty2.default)(this, "AXIOS_TIMEOUT_MS", 10_000);
    (0, _defineProperty2.default)(this, "receiver", void 0);
    (0, _defineProperty2.default)(this, "esClusterInfo", void 0);
    (0, _defineProperty2.default)(this, "cache", new Map());
  }
  async start(receiver, cdn = DEFAULT_CDN_CONFIG) {
    var _this$esClusterInfo, _this$esClusterInfo$v;
    this.receiver = receiver;
    this.esClusterInfo = await this.receiver.fetchClusterInfo();
    this.cdn = cdn;
    if ((_this$esClusterInfo = this.esClusterInfo) !== null && _this$esClusterInfo !== void 0 && (_this$esClusterInfo$v = _this$esClusterInfo.version) !== null && _this$esClusterInfo$v !== void 0 && _this$esClusterInfo$v.number) {
      const version = this.esClusterInfo.version.number.substring(0, this.esClusterInfo.version.number.indexOf('-')) || this.esClusterInfo.version.number;
      this.manifestUrl = `${cdn.url}/downloads/kibana/manifest/artifacts-${version}.zip`;
    }
  }
  async getArtifact(name) {
    return _axios.default.get(this.getManifestUrl(), {
      headers: this.headers(name),
      timeout: this.AXIOS_TIMEOUT_MS,
      validateStatus: status => status < 500,
      responseType: 'arraybuffer'
    }).then(async response => {
      switch (response.status) {
        case 200:
          const manifest = {
            data: await this.getManifest(name, response.data),
            notModified: false
          };
          // only update etag if we got a valid manifest
          if (response.headers && response.headers.etag) {
            var _response$headers$eta, _response$headers;
            const cacheEntry = {
              manifest: {
                ...manifest,
                notModified: true
              },
              etag: (_response$headers$eta = (_response$headers = response.headers) === null || _response$headers === void 0 ? void 0 : _response$headers.etag) !== null && _response$headers$eta !== void 0 ? _response$headers$eta : ''
            };
            this.cache.set(name, cacheEntry);
          }
          return (0, _lodash.cloneDeep)(manifest);
        case 304:
          return (0, _lodash.cloneDeep)(this.getCachedManifest(name));
        case 404:
          // just in case, remove the entry
          this.cache.delete(name);
          throw Error(`No manifest resource found at url: ${this.manifestUrl}`);
        default:
          throw Error(`Failed to download manifest, unexpected status code: ${response.status}`);
      }
    });
  }
  getManifestUrl() {
    if (this.manifestUrl) {
      return this.manifestUrl;
    } else {
      var _this$esClusterInfo2, _this$esClusterInfo2$;
      throw Error(`No manifest url for version ${(_this$esClusterInfo2 = this.esClusterInfo) === null || _this$esClusterInfo2 === void 0 ? void 0 : (_this$esClusterInfo2$ = _this$esClusterInfo2.version) === null || _this$esClusterInfo2$ === void 0 ? void 0 : _this$esClusterInfo2$.number}`);
    }
  }
  getCachedManifest(name) {
    const entry = this.cache.get(name);
    if (!entry) {
      throw Error(`No cached manifest for name ${name}`);
    }
    return entry.manifest;
  }
  async getManifest(name, data) {
    var _manifest$artifacts$n;
    const zip = new _admZip.default(data);
    const manifestFile = zip.getEntries().find(entry => {
      return entry.entryName === 'manifest.json';
    });
    const manifestSigFile = zip.getEntries().find(entry => {
      return entry.entryName === 'manifest.sig';
    });
    if (!manifestFile) {
      throw Error('No manifest.json in artifact zip');
    }
    if (!manifestSigFile) {
      throw Error('No manifest.sig in artifact zip');
    }
    if (!this.isSignatureValid(manifestFile.getData(), manifestSigFile.getData())) {
      throw Error('Invalid manifest signature');
    }
    const manifest = JSON.parse(manifestFile.getData().toString());
    const relativeUrl = (_manifest$artifacts$n = manifest.artifacts[name]) === null || _manifest$artifacts$n === void 0 ? void 0 : _manifest$artifacts$n.relative_url;
    if (relativeUrl) {
      var _this$cdn;
      const url = `${(_this$cdn = this.cdn) === null || _this$cdn === void 0 ? void 0 : _this$cdn.url}${relativeUrl}`;
      const artifactResponse = await _axios.default.get(url, {
        timeout: this.AXIOS_TIMEOUT_MS
      });
      return artifactResponse.data;
    } else {
      throw Error(`No artifact for name ${name}`);
    }
  }

  // morre info https://www.rfc-editor.org/rfc/rfc9110#name-etag
  headers(name) {
    var _this$cache$get;
    const etag = (_this$cache$get = this.cache.get(name)) === null || _this$cache$get === void 0 ? void 0 : _this$cache$get.etag;
    if (etag) {
      return {
        'If-None-Match': etag
      };
    }
    return {};
  }
  isSignatureValid(data, signature) {
    if (!this.cdn) {
      throw Error('No CDN configuration provided');
    }
    const verifier = (0, _crypto.createVerify)('RSA-SHA256');
    verifier.update(data);
    verifier.end();
    return verifier.verify(this.cdn.pubKey, signature);
  }
}
exports.Artifact = Artifact;
const artifactService = exports.artifactService = new Artifact();