"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemMigrationsTaskClient = void 0;
var _constants = require("../../../../../common/siem_migrations/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SiemMigrationsTaskClient {
  constructor(migrationsRunning, logger, data, request, currentUser, dependencies) {
    this.migrationsRunning = migrationsRunning;
    this.logger = logger;
    this.data = data;
    this.request = request;
    this.currentUser = currentUser;
    this.dependencies = dependencies;
  }

  /** Starts a rule migration task */
  async start(params) {
    const {
      migrationId,
      connectorId,
      invocationConfig
    } = params;
    if (this.migrationsRunning.has(migrationId)) {
      return {
        exists: true,
        started: false
      };
    }
    // Just in case some previous execution was interrupted without cleaning up
    await this.data.items.updateStatus(migrationId, {
      status: _constants.SiemMigrationStatus.PROCESSING
    }, _constants.SiemMigrationStatus.PENDING, {
      refresh: true
    });
    const {
      items
    } = await this.data.items.getStats(migrationId);
    if (items.total === 0) {
      return {
        exists: false,
        started: false
      };
    }
    if (items.pending === 0) {
      return {
        exists: true,
        started: false
      };
    }
    const migrationLogger = this.logger.get(migrationId);
    const abortController = new AbortController();
    const migrationTaskRunner = new this.TaskRunnerClass(migrationId, this.request, this.currentUser, abortController, this.data, migrationLogger, this.dependencies);
    await migrationTaskRunner.setup(connectorId);
    if (this.migrationsRunning.has(migrationId)) {
      // Just to prevent a race condition in the setup
      throw new Error('Task already running for this migration');
    }
    migrationLogger.info('Starting migration');
    this.migrationsRunning.set(migrationId, migrationTaskRunner);
    await this.data.migrations.saveAsStarted({
      id: migrationId,
      connectorId,
      ...this.getLastExecutionConfig(invocationConfig)
    });

    // run the migration in the background without awaiting and resolve the `start` promise
    migrationTaskRunner.run(invocationConfig).then(() => {
      // The task runner has finished normally. Abort errors are also handled here, it's an expected finish scenario, nothing special should be done.
      migrationLogger.debug('Migration execution task finished');
      this.data.migrations.saveAsFinished({
        id: migrationId
      }).catch(error => {
        migrationLogger.error(`Error saving migration as finished: ${error}`);
      });
    }).catch(error => {
      // Unexpected errors, no use in throwing them since the `start` promise is long gone. Just log and store the error message
      migrationLogger.error(`Error executing migration task: ${error}`);
      this.data.migrations.saveAsFailed({
        id: migrationId,
        error: error.message
      }).catch(saveError => {
        migrationLogger.error(`Error saving migration as failed: ${saveError}`);
      });
    }).finally(() => {
      this.migrationsRunning.delete(migrationId);
    });
    return {
      exists: true,
      started: true
    };
  }

  /** Updates all the rules in a migration to be re-executed */
  async updateToRetry(migrationId, filter) {
    this.logger.warn(`Updating migration ID:${migrationId} to retry with filter: ${JSON.stringify(filter, null, 2)}`);
    if (this.migrationsRunning.has(migrationId)) {
      // not update migrations that are currently running
      return {
        updated: false
      };
    }
    filter.installed = false; // only retry rules that are not installed
    await this.data.items.updateStatus(migrationId, filter, _constants.SiemMigrationStatus.PENDING, {
      refresh: true
    });
    return {
      updated: true
    };
  }

  /** Returns the stats of a migration */
  async getStats(migrationId) {
    const migration = await this.data.migrations.get(migrationId);
    if (!migration) {
      throw new Error(`Migration with ID ${migrationId} not found`);
    }
    const dataStats = await this.data.items.getStats(migrationId);
    const taskStats = this.getTaskStats(migration, dataStats.items);
    return {
      ...taskStats,
      ...dataStats,
      name: migration.name
    };
  }

  /** Returns the stats of all migrations */
  async getAllStats() {
    const allDataStats = await this.data.items.getAllStats();
    const allMigrations = await this.data.migrations.getAll();
    const allMigrationsMap = new Map(allMigrations.map(migration => [migration.id, migration]));
    const allStats = [];
    for (const dataStats of allDataStats) {
      const migration = allMigrationsMap.get(dataStats.id);
      if (migration) {
        const tasksStats = this.getTaskStats(migration, dataStats.items);
        allStats.push({
          name: migration.name,
          ...tasksStats,
          ...dataStats
        });
      }
    }
    return allStats;
  }
  getTaskStats(migration, dataStats) {
    return {
      status: this.getTaskStatus(migration, dataStats),
      last_execution: migration.last_execution
    };
  }
  getTaskStatus(migration, dataStats) {
    const {
      id: migrationId,
      last_execution: lastExecution
    } = migration;
    if (this.migrationsRunning.has(migrationId)) {
      return _constants.SiemMigrationTaskStatus.RUNNING;
    }
    if (dataStats.completed + dataStats.failed === dataStats.total) {
      return _constants.SiemMigrationTaskStatus.FINISHED;
    }
    if (lastExecution !== null && lastExecution !== void 0 && lastExecution.is_stopped) {
      return _constants.SiemMigrationTaskStatus.STOPPED;
    }
    if (dataStats.pending === dataStats.total) {
      return _constants.SiemMigrationTaskStatus.READY;
    }
    return _constants.SiemMigrationTaskStatus.INTERRUPTED;
  }

  // Overridable method to get the last execution config
  getLastExecutionConfig(_invocationConfig) {
    return {};
  }

  /** Stops one running migration */
  async stop(migrationId) {
    try {
      const migrationRunning = this.migrationsRunning.get(migrationId);
      if (migrationRunning) {
        migrationRunning.abortController.abort('Stopped by user');
        await this.data.migrations.setIsStopped({
          id: migrationId
        });
        return {
          exists: true,
          stopped: true
        };
      }
      const {
        items
      } = await this.data.items.getStats(migrationId);
      if (items.total > 0) {
        return {
          exists: true,
          stopped: true
        };
      }
      return {
        exists: false,
        stopped: true
      };
    } catch (err) {
      this.logger.error(`Error stopping migration ID:${migrationId}`, err);
      return {
        exists: true,
        stopped: false
      };
    }
  }

  /** Creates a new evaluator for the rule migration task */
  async evaluate(params) {
    if (!this.EvaluatorClass) {
      throw new Error('Evaluator class needs to be defined to use evaluate method');
    }
    const {
      evaluationId,
      langsmithOptions,
      connectorId,
      abortController
    } = params;
    const migrationLogger = this.logger.get('evaluate');
    const taskRunner = new this.TaskRunnerClass(evaluationId, this.request, this.currentUser, abortController, this.data, migrationLogger, this.dependencies);
    const migrationTaskEvaluator = new this.EvaluatorClass(taskRunner, this.dependencies, this.logger);
    await migrationTaskEvaluator.evaluate({
      connectorId,
      langsmithOptions
    });
  }

  /** Returns if a migration is running or not */
  isMigrationRunning(migrationId) {
    return this.migrationsRunning.has(migrationId);
  }
}
exports.SiemMigrationsTaskClient = SiemMigrationsTaskClient;