"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResourceRetriever = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _types = require("../../../../../../common/siem_migrations/rules/resources/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ResourceRetriever {
  constructor(migrationId, resourcesDataClient, ResourceIdentifier) {
    (0, _defineProperty2.default)(this, "existingResources", void 0);
    this.migrationId = migrationId;
    this.resourcesDataClient = resourcesDataClient;
    this.ResourceIdentifier = ResourceIdentifier;
  }
  async initialize() {
    const batches = this.resourcesDataClient.searchBatches(this.migrationId, {
      filters: {
        hasContent: true
      }
    } // filters out missing (undefined) content resources, empty strings content will be included
    );
    const existingResources = {
      macro: {},
      lookup: {}
    };
    let resources;
    do {
      resources = await batches.next();
      resources.forEach(resource => {
        // currenctly only supported for splunk
        if (resource.type === 'macro' || resource.type === 'lookup') {
          existingResources[resource.type][resource.name] = resource;
        }
      });
    } while (resources.length > 0);
    this.existingResources = existingResources;
  }
  async getResources(originalItem) {
    if (!(0, _types.isResourceSupportedVendor)(originalItem.vendor)) {
      throw new Error(`Resource retrievel is not supported for vendor: ${originalItem.vendor}`);
    }
    const existingResources = this.existingResources;
    if (!existingResources) {
      throw new Error('initialize must be called before calling getResources');
    }
    const resourceIdentifier = new this.ResourceIdentifier(originalItem.vendor);
    const resourcesIdentifiedFromRule = await resourceIdentifier.fromOriginal(originalItem);
    const macrosFound = new Map();
    const lookupsFound = new Map();
    resourcesIdentifiedFromRule.forEach(resource => {
      if (resource.type !== 'macro' && resource.type !== 'lookup') {
        return;
      }
      const existingResource = existingResources[resource.type][resource.name];
      if (existingResource) {
        if (resource.type === 'macro') {
          macrosFound.set(resource.name, existingResource);
        } else if (resource.type === 'lookup') {
          lookupsFound.set(resource.name, existingResource);
        }
      }
    });
    const resourcesFound = [...macrosFound.values(), ...lookupsFound.values()];
    if (!resourcesFound.length) {
      return {};
    }
    let nestedResourcesFound = resourcesFound;
    do {
      const nestedResourcesIdentified = resourceIdentifier.fromResources(nestedResourcesFound);
      nestedResourcesFound = [];
      nestedResourcesIdentified.forEach(resource => {
        if (resource.type !== 'macro' && resource.type !== 'lookup') {
          return;
        }
        const existingResource = existingResources[resource.type][resource.name];
        if (existingResource) {
          nestedResourcesFound.push(existingResource);
          if (resource.type === 'macro') {
            macrosFound.set(resource.name, existingResource);
          } else if (resource.type === 'lookup') {
            lookupsFound.set(resource.name, existingResource);
          }
        }
      });
    } while (nestedResourcesFound.length > 0);
    return {
      ...(macrosFound.size > 0 ? {
        macro: this.formatOutput(macrosFound)
      } : {}),
      ...(lookupsFound.size > 0 ? {
        lookup: this.formatOutput(lookupsFound)
      } : {})
    };
  }
  formatOutput(resources) {
    return Array.from(resources.values()).map(({
      name,
      content,
      type
    }) => ({
      name,
      content,
      type
    }));
  }
}
exports.ResourceRetriever = ResourceRetriever;