"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerApiAccessControl = void 0;
var _actions = require("@kbn/security-solution-features/actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// The `securitySolutionProductFeature:` prefix is used for ProductFeature based control.
// Should be used only by routes that do not need RBAC, only direct productFeature control.
const APP_FEATURE_TAG_PREFIX = 'securitySolutionProductFeature:';
const isAuthzEnabled = authz => {
  return Boolean(authz === null || authz === void 0 ? void 0 : authz.requiredPrivileges);
};

/**
 * Registers a route access control to ensure that the product features are enabled for the route.
 * Specially required for superuser (`*`) roles with universal access to all APIs.
 * This control checks two things:
 * - `securitySolutionProductFeature:` tag: verifies if the corresponding product feature is enabled.
 * - `requiredPrivileges` in the route's authz config: checks if the required privileges are enabled.
 */
const registerApiAccessControl = (service, http) => {
  /** Returns true only if the API privilege is a security action and is disabled */
  const isApiPrivilegeSecurityAndDisabled = apiPrivilege => {
    if (apiPrivilege.startsWith(_actions.API_ACTION_PREFIX)) {
      return !service.isActionRegistered(`api:${apiPrivilege}`);
    }
    return false;
  };
  http.registerOnPostAuth((request, response, toolkit) => {
    var _request$route$option2;
    for (const tag of (_request$route$option = request.route.options.tags) !== null && _request$route$option !== void 0 ? _request$route$option : []) {
      var _request$route$option;
      let isEnabled = true;
      if (tag.startsWith(APP_FEATURE_TAG_PREFIX)) {
        isEnabled = service.isEnabled(tag.substring(APP_FEATURE_TAG_PREFIX.length));
      }
      if (!isEnabled) {
        service.logger.warn(`Accessing disabled route "${request.url.pathname}${request.url.search}": responding with 404`);
        return response.notFound();
      }
    }

    // This control ensures the action privileges have been registered by the productFeature service,
    // preventing full access (`*`) roles, such as superuser, from bypassing productFeature controls.
    const authz = (_request$route$option2 = request.route.options.security) === null || _request$route$option2 === void 0 ? void 0 : _request$route$option2.authz;
    if (isAuthzEnabled(authz)) {
      const disabled = authz.requiredPrivileges.some(privilegeEntry => {
        if (typeof privilegeEntry === 'object') {
          if (privilegeEntry.allRequired) {
            if (privilegeEntry.allRequired.some(entry => typeof entry === 'string' ? isApiPrivilegeSecurityAndDisabled(entry) : entry.anyOf.every(isApiPrivilegeSecurityAndDisabled))) {
              return true;
            }
          }
          if (privilegeEntry.anyRequired) {
            if (privilegeEntry.anyRequired.every(entry => typeof entry === 'string' ? isApiPrivilegeSecurityAndDisabled(entry) : entry.allOf.some(isApiPrivilegeSecurityAndDisabled))) {
              return true;
            }
          }
          return false;
        } else {
          return isApiPrivilegeSecurityAndDisabled(privilegeEntry);
        }
      });
      if (disabled) {
        service.logger.warn(`Accessing disabled route "${request.url.pathname}${request.url.search}": responding with 404`);
        return response.notFound();
      }
    }
    return toolkit.next();
  });
};
exports.registerApiAccessControl = registerApiAccessControl;