"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getESQL = exports.getCompositeQuery = exports.calculateScoresWithESQL = exports.buildRiskScoreBucket = void 0;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _technical_rule_data_field_names = require("@kbn/rule-registry-plugin/common/technical_rule_data_field_names");
var _Record = require("fp-ts/Record");
var _types = require("../../../../common/entity_analytics/types");
var _utils = require("../../../../common/entity_analytics/utils");
var _with_security_span = require("../../../utils/with_security_span");
var _constants = require("./constants");
var _calculate_risk_scores = require("./calculate_risk_scores");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const calculateScoresWithESQL = async params => (0, _with_security_span.withSecuritySpan)('calculateRiskScores', async () => {
  const {
    identifierType,
    logger,
    esClient
  } = params;
  const now = new Date().toISOString();
  const identifierTypes = identifierType ? [identifierType] : (0, _utils.getEntityAnalyticsEntityTypes)();

  // Create separate queries for each entity type with entity-specific filters
  const entityQueries = identifierTypes.map(entityType => {
    const filter = getFilters(params, entityType);
    return {
      entityType,
      query: getCompositeQuery([entityType], filter, params)
    };
  });
  logger.trace(`STEP ONE: Executing ESQL Risk Score queries for entity types: ${identifierTypes.join(', ')}`);

  // Execute queries for each entity type
  const responses = await Promise.all(entityQueries.map(async ({
    entityType,
    query
  }) => {
    logger.trace(`Executing ESQL Risk Score query for ${entityType}:\n${JSON.stringify(query)}`);
    let error = null;
    const response = await esClient.search(query).catch(e => {
      logger.error(`Error executing composite query for ${entityType}: ${e.message}`);
      error = e;
      return null;
    });
    return {
      entityType,
      response,
      query,
      error
    };
  }));

  // Combine results from all entity queries
  const combinedAggregations = {};
  responses.forEach(({
    entityType,
    response
  }) => {
    if (response !== null && response !== void 0 && response.aggregations && response.aggregations[entityType]) {
      combinedAggregations[entityType] = response.aggregations[entityType];
    }
  });

  // Check if all queries that had errors failed due to index_not_found_exception
  const errorsPresent = responses.filter(({
    error
  }) => error).length;
  const indexNotFoundErrors = responses.filter(({
    error
  }) => {
    if (!error) return false;
    const errorMessage = error instanceof Error ? error.message : String(error);
    return errorMessage.includes('index_not_found_exception') || errorMessage.includes('no such index') || errorMessage.includes('NoShardAvailableActionException');
  }).length;

  // If we have no aggregations, return empty scores if:
  // 1. All queries that had errors were index-not-found errors
  // 2. OR there were no errors at all (valid index pattern with no data)
  const shouldReturnEmptyScores = errorsPresent === 0 || errorsPresent > 0 && errorsPresent === indexNotFoundErrors;
  if (Object.keys(combinedAggregations).length === 0) {
    if (shouldReturnEmptyScores) {
      return {
        after_keys: {},
        scores: {
          host: [],
          user: [],
          service: []
        }
      };
    }
    // Log the actual errors for debugging
    responses.forEach(({
      entityType,
      error
    }) => {
      if (error) {
        logger.error(`Query failed for ${entityType}: ${error instanceof Error ? error.message : String(error)}`);
      }
    });
    // Otherwise, throw an error as before
    throw new Error('No aggregations in any composite response');
  }
  const promises = (0, _Record.toEntries)(combinedAggregations).map(async ([entityType, aggregationData]) => {
    var _entityType;
    const {
      buckets,
      after_key: afterKey
    } = aggregationData;
    const entities = buckets.map(({
      key
    }) => key[_types.EntityTypeToIdentifierField[entityType]]);
    if (entities.length === 0) {
      return Promise.resolve([entityType, {
        afterKey: afterKey || {},
        scores: []
      }]);
    }
    const bounds = {
      lower: (_entityType = params.afterKeys[entityType]) === null || _entityType === void 0 ? void 0 : _entityType[_types.EntityTypeToIdentifierField[entityType]],
      upper: afterKey === null || afterKey === void 0 ? void 0 : afterKey[_types.EntityTypeToIdentifierField[entityType]]
    };
    const query = getESQL(entityType, bounds, params.alertSampleSizePerShard || 10000, params.pageSize, params.index);
    const entityFilter = getFilters(params, entityType);
    return esClient.esql.query({
      query,
      filter: {
        bool: {
          filter: entityFilter
        }
      }
    }).then(rs => rs.values.map(buildRiskScoreBucket(entityType, params.index))).then(riskScoreBuckets => {
      return (0, _calculate_risk_scores.processScores)({
        assetCriticalityService: params.assetCriticalityService,
        buckets: riskScoreBuckets,
        identifierField: _types.EntityTypeToIdentifierField[entityType],
        logger,
        now,
        identifierType: entityType,
        weights: params.weights
      });
    }).then(scores => {
      return [entityType, {
        scores,
        afterKey: afterKey
      }];
    }).catch(error => {
      logger.error(`Error executing ESQL query for entity type ${entityType}: ${error.message}`);
      logger.error(`Query: ${query}`);
      return [entityType, {
        afterKey: afterKey || {},
        scores: []
      }];
    });
  });
  const esqlResults = await Promise.all(promises);
  const results = esqlResults.reduce((res, [entityType, {
    afterKey,
    scores
  }]) => {
    res.after_keys[entityType] = afterKey || {};
    res.scores[entityType] = scores;
    return res;
  }, {
    after_keys: {},
    scores: {}
  });
  return results;
});
exports.calculateScoresWithESQL = calculateScoresWithESQL;
const getFilters = (options, entityType) => {
  const {
    excludeAlertStatuses = [],
    excludeAlertTags = [],
    range,
    filter: userFilter,
    filters: customFilters
  } = options;
  const filters = [(0, _calculate_risk_scores.filterFromRange)(range), {
    exists: {
      field: _technical_rule_data_field_names.ALERT_RISK_SCORE
    }
  }];
  if (excludeAlertStatuses.length > 0) {
    filters.push({
      bool: {
        must_not: {
          terms: {
            [_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS]: excludeAlertStatuses
          }
        }
      }
    });
  }
  if (!(0, _lodash.isEmpty)(userFilter)) {
    filters.push(userFilter);
  }
  if (excludeAlertTags.length > 0) {
    filters.push({
      bool: {
        must_not: {
          terms: {
            [_technical_rule_data_field_names.ALERT_WORKFLOW_TAGS]: excludeAlertTags
          }
        }
      }
    });
  }

  // Apply entity-specific custom filters (EXCLUSIVE - exclude matching alerts)
  if (customFilters && customFilters.length > 0 && entityType) {
    customFilters.filter(customFilter => customFilter.entity_types.includes(entityType)).forEach(customFilter => {
      try {
        const kqlQuery = (0, _esQuery.fromKueryExpression)(customFilter.filter);
        const esQuery = (0, _esQuery.toElasticsearchQuery)(kqlQuery);
        if (esQuery) {
          filters.push({
            bool: {
              must: esQuery
            }
          });
        }
      } catch (error) {
        // Silently ignore invalid KQL filters to prevent query failures
      }
    });
  }
  return filters;
};
const getCompositeQuery = (entityTypes, filter, params) => {
  return {
    size: 0,
    index: params.index,
    ignore_unavailable: true,
    runtime_mappings: params.runtimeMappings,
    query: {
      function_score: {
        query: {
          bool: {
            filter,
            should: [{
              match_all: {} // This forces ES to calculate score
            }]
          }
        },
        field_value_factor: {
          field: _technical_rule_data_field_names.ALERT_RISK_SCORE // sort by risk score
        }
      }
    },
    aggs: entityTypes.reduce((aggs, entityType) => {
      const idField = _types.EntityTypeToIdentifierField[entityType];
      return {
        ...aggs,
        [entityType]: {
          composite: {
            size: params.pageSize,
            sources: [{
              [idField]: {
                terms: {
                  field: idField
                }
              }
            }],
            after: params.afterKeys[entityType]
          }
        }
      };
    }, {})
  };
};
exports.getCompositeQuery = getCompositeQuery;
const getESQL = (entityType, afterKeys, sampleSize, pageSize, index = '.alerts-security.alerts-default') => {
  const identifierField = _types.EntityTypeToIdentifierField[entityType];
  const lower = afterKeys.lower ? `${identifierField} > ${afterKeys.lower}` : undefined;
  const upper = afterKeys.upper ? `${identifierField} <= ${afterKeys.upper}` : undefined;
  if (!lower && !upper) {
    throw new Error('Either lower or upper after key must be provided for pagination');
  }
  const rangeClause = [lower, upper].filter(Boolean).join(' and ');
  const query = /* SQL */`
  FROM ${index} METADATA _index
    | WHERE kibana.alert.risk_score IS NOT NULL AND KQL("${rangeClause}")
    | RENAME kibana.alert.risk_score as risk_score,
             kibana.alert.rule.name as rule_name,
             kibana.alert.rule.uuid as rule_id,
             kibana.alert.uuid as alert_id,
             event.kind as category,
             @timestamp as time
    | EVAL input = CONCAT(""" {"risk_score": """", risk_score::keyword, """", "time": """", time::keyword, """", "index": """", _index, """", "rule_name": """", rule_name, """\", "category": """", category, """\", "id": \"""", alert_id, """\" } """)
    | STATS
        alert_count = count(risk_score),
        scores = MV_PSERIES_WEIGHTED_SUM(TOP(risk_score, ${sampleSize}, "desc"), ${_constants.RIEMANN_ZETA_S_VALUE}),
        risk_inputs = TOP(input, 10, "desc")
    BY ${identifierField}
    | SORT scores DESC
    | LIMIT ${pageSize}
  `;
  return query;
};
exports.getESQL = getESQL;
const buildRiskScoreBucket = (entityType, index) => row => {
  const [count, score, _inputs, entity] = row;
  const inputs = (Array.isArray(_inputs) ? _inputs : [_inputs]).map((input, i) => {
    const parsedRiskInputData = JSON.parse(input);
    const value = parseFloat(parsedRiskInputData.risk_score);
    const currentScore = value / Math.pow(i + 1, _constants.RIEMANN_ZETA_S_VALUE);
    const {
      risk_score: _,
      ...otherFields
    } = parsedRiskInputData;
    return {
      ...otherFields,
      score: value,
      contribution: currentScore / _constants.RIEMANN_ZETA_VALUE,
      index
    };
  });
  return {
    key: {
      [_types.EntityTypeToIdentifierField[entityType]]: entity
    },
    doc_count: count,
    top_inputs: {
      doc_count: inputs.length,
      risk_details: {
        value: {
          score,
          normalized_score: score / _constants.RIEMANN_ZETA_VALUE,
          // normalize value to be between 0-100
          notes: [],
          category_1_score: score,
          // Don't normalize here - will be normalized in calculate_risk_scores.ts
          category_1_count: count,
          risk_inputs: inputs
        }
      }
    }
  };
};
exports.buildRiskScoreBucket = buildRiskScoreBucket;