"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPatternMatcherService = void 0;
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _queries = require("./queries");
var _search = require("../../../../users/search");
var _generate_monitoring_labels = require("../../generate_monitoring_labels");
var _sync_markers = require("../sync_markers/sync_markers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isTimestampGreaterThan = (date1, date2) => {
  const m1 = (0, _moment.default)(date1);
  const m2 = (0, _moment.default)(date2);
  if (!m1.isValid()) return false;
  if (!m2.isValid()) return true;
  return m1.isAfter(m2);
};
const createPatternMatcherService = (dataClient, soClient) => {
  const searchService = (0, _search.createSearchService)(dataClient);
  const syncMarkerService = (0, _sync_markers.createSyncMarkersService)(dataClient, soClient);
  const findPrivilegedUsersFromMatchers = async source => {
    var _source$matchers;
    /**
     * Empty matchers policy: SAFE DEFAULT
     * If no matchers are configured, we *cannot* infer privileged users.
     * Return none and log a warning so UI can prompt configuration.
     */
    if (!((_source$matchers = source.matchers) !== null && _source$matchers !== void 0 && _source$matchers.length)) {
      var _source$id;
      dataClient.log('info', `No matchers for source id=${(_source$id = source.id) !== null && _source$id !== void 0 ? _source$id : '(unknown)'}. Returning 0 privileged users`);
      return [];
    }
    const esClient = dataClient.deps.clusterClient.asCurrentUser;
    // the last processed user from previous task run.
    const lastProcessedTimeStamp = await syncMarkerService.getLastProcessedMarker(source);
    let afterKey;
    let fetchMore = true;
    const pageSize = 100; // number of agg buckets per page
    const users = [];

    // In the search should be ALL since last task sync, not just this run.
    let maxProcessedTimeStamp = lastProcessedTimeStamp; // latest seen during THIS run of the task.

    try {
      while (fetchMore) {
        var _response$aggregation, _privUserAgg$buckets;
        const response = await esClient.search({
          index: source.indexPattern,
          ...(0, _queries.buildPrivilegedSearchBody)(source.matchers, lastProcessedTimeStamp, afterKey, pageSize)
        });
        const aggregations = response.aggregations;
        const privUserAgg = (_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : _response$aggregation.privileged_user_status_since_last_run;
        const buckets = (_privUserAgg$buckets = privUserAgg === null || privUserAgg === void 0 ? void 0 : privUserAgg.buckets) !== null && _privUserAgg$buckets !== void 0 ? _privUserAgg$buckets : [];
        if (buckets.length && aggregations) {
          const {
            users: privMonUsers,
            maxTimestamp
          } = await parseAggregationResponse(aggregations, source);
          // update running max timestamp seen
          maxProcessedTimeStamp = maxTimestamp !== null && maxTimestamp !== void 0 ? maxTimestamp : maxProcessedTimeStamp;
          users.push(...privMonUsers);
        }

        // next page
        afterKey = privUserAgg === null || privUserAgg === void 0 ? void 0 : privUserAgg.after_key;
        fetchMore = Boolean(afterKey);
      }
      dataClient.log('info', `Found ${users.length} privileged users from matchers.`);
      await syncMarkerService.updateLastProcessedMarker(source, maxProcessedTimeStamp);
      return users;
    } catch (error) {
      dataClient.log('error', `Error finding privileged users from matchers: ${error.message}`);
      return [];
    }
  };
  const parseAggregationResponse = async (aggregation, source) => {
    var _aggregation$privileg;
    const buckets = (_aggregation$privileg = aggregation.privileged_user_status_since_last_run) === null || _aggregation$privileg === void 0 ? void 0 : _aggregation$privileg.buckets;
    if (!buckets) {
      return {
        users: []
      };
    }
    const batchUsernames = buckets.map(bucket => bucket.key.username);
    const existingUserMap = await searchService.getExistingUsersMap((0, _lodash.uniq)(batchUsernames));
    let maxTimestamp;
    const users = buckets.map(bucket => {
      var _bucket$latest_doc_fo, _bucket$latest_doc_fo2, _bucket$latest_doc_fo3, _ref, _topHit$fields$userI, _topHit$fields, _topHit$fields$userI2, _topHit$_source, _topHit$_source$user, _topHit$_source2;
      const topHit = (_bucket$latest_doc_fo = bucket.latest_doc_for_user) === null || _bucket$latest_doc_fo === void 0 ? void 0 : (_bucket$latest_doc_fo2 = _bucket$latest_doc_fo.hits) === null || _bucket$latest_doc_fo2 === void 0 ? void 0 : (_bucket$latest_doc_fo3 = _bucket$latest_doc_fo2.hits) === null || _bucket$latest_doc_fo3 === void 0 ? void 0 : _bucket$latest_doc_fo3[0];
      const isPrivileged = Boolean((_ref = (_topHit$fields$userI = topHit === null || topHit === void 0 ? void 0 : (_topHit$fields = topHit.fields) === null || _topHit$fields === void 0 ? void 0 : (_topHit$fields$userI2 = _topHit$fields['user.is_privileged']) === null || _topHit$fields$userI2 === void 0 ? void 0 : _topHit$fields$userI2[0]) !== null && _topHit$fields$userI !== void 0 ? _topHit$fields$userI : topHit === null || topHit === void 0 ? void 0 : (_topHit$_source = topHit._source) === null || _topHit$_source === void 0 ? void 0 : (_topHit$_source$user = _topHit$_source.user) === null || _topHit$_source$user === void 0 ? void 0 : _topHit$_source$user.is_privileged) !== null && _ref !== void 0 ? _ref : false);
      const timestamp = topHit === null || topHit === void 0 ? void 0 : (_topHit$_source2 = topHit._source) === null || _topHit$_source2 === void 0 ? void 0 : _topHit$_source2['@timestamp'];
      if (timestamp) {
        if (!maxTimestamp) maxTimestamp = timestamp;else if (isTimestampGreaterThan(timestamp, maxTimestamp)) maxTimestamp = timestamp;
      }
      const monitoringLabels = (0, _generate_monitoring_labels.generateMonitoringLabels)(source.id, source.matchers, topHit._source || {});
      return {
        sourceId: source.id,
        username: bucket.key.username,
        existingUserId: existingUserMap.get(bucket.key.username),
        isPrivileged,
        monitoringLabels
      };
    });
    return {
      users,
      maxTimestamp
    };
  };
  return {
    findPrivilegedUsersFromMatchers
  };
};
exports.createPatternMatcherService = createPatternMatcherService;