"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EntityStoreDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _entity_client = require("@kbn/entityManager-plugin/server/lib/entity_client");
var _fp = require("lodash/fp");
var _moment = _interopRequireDefault(require("moment"));
var _saved_objects = require("@kbn/entityManager-plugin/server/saved_objects");
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _constants = require("../../../../common/constants");
var _constants2 = require("../../../../common/entity_analytics/entity_store/constants");
var _utils = require("../../../../common/entity_analytics/utils");
var _privileges = require("../../../../common/entity_analytics/privileges");
var _merge = require("../../../../common/utils/objects/merge");
var _types = require("../../../../common/entity_analytics/types");
var _entity_analytics = require("../../../../common/api/entity_analytics");
var _engine_descriptor = require("./saved_object/engine_descriptor");
var _constants3 = require("./constants");
var _asset_criticality_migration_client = require("../asset_criticality/asset_criticality_migration_client");
var _tasks = require("./tasks");
var _elasticsearch_assets = require("./elasticsearch_assets");
var _risk_score_data_client = require("../risk_score/risk_score_data_client");
var _utils2 = require("./utils");
var _actions = require("./auditing/actions");
var _audit = require("../audit");
var _events = require("../../telemetry/event_based/events");
var _constants4 = require("../asset_criticality/constants");
var _engine_description = require("./installation/engine_description");
var _entity_manager_conversion = require("./entity_definitions/entity_manager_conversion");
var _check_and_format_privileges = require("../utils/check_and_format_privileges");
var _saved_object = require("./saved_object");
var _updates_entity_data_stream = require("./elasticsearch_assets/updates_entity_data_stream");
var _ilm_policy_status = require("./elasticsearch_assets/ilm_policy_status");
var _updates_component_template = require("./elasticsearch_assets/updates_component_template");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Workaround. TransformState type is wrong. The health type should be: TransformHealth from '@kbn/transform-plugin/common/types/transform_stats'

class EntityStoreDataClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "engineClient", void 0);
    (0, _defineProperty2.default)(this, "assetCriticalityMigrationClient", void 0);
    (0, _defineProperty2.default)(this, "entityClient", void 0);
    (0, _defineProperty2.default)(this, "riskScoreDataClient", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "apiKeyGenerator", void 0);
    (0, _defineProperty2.default)(this, "uiSettingsClient", void 0);
    (0, _defineProperty2.default)(this, "isServerless", void 0);
    /**
     * Get the index privileges required for installing all entity store resources
     */
    (0, _defineProperty2.default)(this, "getEntityStoreInitPrivileges", async indices => {
      const security = this.options.security;

      // The entity store needs access to all security solution indices
      const indicesPrivileges = this.getEntityStoreSourceRequiredIndicesPrivileges(indices);

      // The entity store has to create the following indices
      indicesPrivileges[_constants2.ENTITY_STORE_INDEX_PATTERN] = ['read', 'manage'];
      indicesPrivileges[_constants2.ENTITY_STORE_UPDATES_INDEX_PATTERN] = ['read', 'manage'];
      indicesPrivileges[_constants.RISK_SCORE_INDEX_PATTERN] = ['read', 'manage'];
      indicesPrivileges[_constants2.ENTITY_STORE_HISTORY_INDEX_PATTERN] = ['create_index', 'manage', 'read', 'write'];
      return (0, _check_and_format_privileges.checkAndFormatPrivileges)({
        request: this.options.request,
        security,
        privilegesToCheck: {
          kibana: [security.authz.actions.savedObject.get(_saved_object.entityEngineDescriptorTypeName, 'create'), security.authz.actions.savedObject.get(_saved_objects.SO_ENTITY_DEFINITION_TYPE, 'create')],
          elasticsearch: {
            cluster: _constants2.ENTITY_STORE_REQUIRED_ES_CLUSTER_PRIVILEGES,
            index: indicesPrivileges
          }
        }
      });
    });
    /**
     * Get the index privileges required for running the transform
     */
    (0, _defineProperty2.default)(this, "getEntityStoreSourceIndicesPrivileges", indexPatterns => {
      const requiredIndicesPrivileges = this.getEntityStoreSourceRequiredIndicesPrivileges(indexPatterns);
      return (0, _check_and_format_privileges.checkAndFormatPrivileges)({
        request: this.options.request,
        security: this.options.security,
        privilegesToCheck: {
          elasticsearch: {
            cluster: [],
            index: requiredIndicesPrivileges
          }
        }
      });
    });
    this.options = options;
    const {
      clusterClient,
      logger,
      soClient,
      auditLogger,
      kibanaVersion,
      namespace,
      apiKeyManager,
      uiSettingsClient,
      isServerless
    } = options;
    this.esClient = clusterClient.asCurrentUser;
    this.apiKeyGenerator = apiKeyManager;
    this.uiSettingsClient = uiSettingsClient;
    this.isServerless = isServerless;
    this.entityClient = new _entity_client.EntityClient({
      clusterClient,
      soClient,
      isServerless,
      logger
    });
    this.engineClient = new _engine_descriptor.EngineDescriptorClient({
      soClient,
      namespace
    });
    this.assetCriticalityMigrationClient = new _asset_criticality_migration_client.AssetCriticalityMigrationClient({
      esClient: this.esClient,
      logger,
      auditLogger
    });
    this.riskScoreDataClient = new _risk_score_data_client.RiskScoreDataClient({
      soClient,
      esClient: this.esClient,
      logger,
      namespace,
      kibanaVersion
    });
  }
  async getEngineComponentsState(type, definition) {
    const {
      namespace,
      taskManager
    } = this.options;
    return definition ? Promise.all([...(taskManager ? [(0, _tasks.getEntityStoreFieldRetentionEnrichTaskState)({
      namespace,
      taskManager
    })] : []), ...(taskManager ? [(0, _tasks.getEntityStoreDataViewRefreshTaskState)({
      namespace,
      taskManager
    })] : []), ...(taskManager ? [(0, _tasks.getEntityStoreSnapshotTaskState)({
      namespace,
      entityType: type,
      taskManager
    })] : []), (0, _elasticsearch_assets.getPlatformPipelineStatus)({
      engineId: definition.id,
      esClient: this.esClient
    }), (0, _elasticsearch_assets.getFieldRetentionEnrichPolicyStatus)({
      definitionMetadata: {
        namespace,
        entityType: type,
        version: definition.version
      },
      esClient: this.esClient
    }), (0, _elasticsearch_assets.getEntityIndexStatus)({
      entityType: type,
      esClient: this.esClient,
      namespace
    }), (0, _elasticsearch_assets.getEntityResetIndexStatus)({
      entityType: type,
      esClient: this.esClient,
      namespace
    }), (0, _updates_entity_data_stream.getEntityUpdatesDataStreamStatus)(type, this.esClient, namespace), ...(await (0, _ilm_policy_status.getEntityILMPolicyStatuses)({
      esClient: this.esClient,
      isServerless: this.isServerless
    })), (0, _elasticsearch_assets.getEntityIndexComponentTemplateStatus)({
      definitionId: definition.id,
      esClient: this.esClient
    }), (0, _updates_component_template.getEntityUpdatesIndexComponentTemplateStatus)(definition.id, this.esClient)]) : Promise.resolve([]);
  }
  async isEngineRunning(type) {
    const engine = await this.engineClient.maybeGet(type);
    return (engine === null || engine === void 0 ? void 0 : engine.status) === _constants3.ENGINE_STATUS.STARTED;
  }
  async enable(requestBodyOverrides, {
    pipelineDebugMode = false
  } = {}) {
    if (!this.options.taskManager) {
      throw new Error('Task Manager is not available');
    }

    // Immediately defer the initialization to the next tick. This way we don't block on the init preflight checks
    const run = fn => new Promise(resolve => setTimeout(() => fn().then(resolve), 0));
    const enabledEntityTypes = await this.getEnabledEntityTypes();

    // When entityTypes param is defined it only enables the engines that are provided
    const enginesTypes = requestBodyOverrides.entityTypes ? requestBodyOverrides.entityTypes.filter(type => enabledEntityTypes.includes(type)) : enabledEntityTypes;
    const promises = enginesTypes.map(entity => run(() => this.init(entity, requestBodyOverrides, {
      pipelineDebugMode
    })));
    const engines = await Promise.all(promises);
    return {
      engines,
      succeeded: true
    };
  }
  async getEnabledEntityTypes() {
    const genericEntityStoreEnabled = await this.uiSettingsClient.get(_managementSettingsIds.SECURITY_SOLUTION_ENABLE_ASSET_INVENTORY_SETTING);
    return (0, _utils.getEnabledEntityTypes)(genericEntityStoreEnabled);
  }
  async status({
    include_components: withComponents = false
  }) {
    const {
      namespace
    } = this.options;
    const {
      engines
    } = await this.engineClient.list();
    const enabledEntityTypes = await this.getEnabledEntityTypes();
    const enabledEngines = engines.filter(engine => {
      return enabledEntityTypes.indexOf(_types.EntityType[engine.type]) > -1;
    });
    let status = _constants3.ENTITY_STORE_STATUS.RUNNING;
    if (enabledEngines.length === 0) {
      status = _constants3.ENTITY_STORE_STATUS.NOT_INSTALLED;
    } else if (enabledEngines.some(engine => engine.status === _constants3.ENGINE_STATUS.ERROR)) {
      status = _constants3.ENTITY_STORE_STATUS.ERROR;
    } else if (enabledEngines.every(engine => engine.status === _constants3.ENGINE_STATUS.STOPPED)) {
      status = _constants3.ENTITY_STORE_STATUS.STOPPED;
    } else if (enabledEngines.some(engine => engine.status === _constants3.ENGINE_STATUS.INSTALLING)) {
      status = _constants3.ENTITY_STORE_STATUS.INSTALLING;
    }
    if (withComponents) {
      const enginesWithComponents = await Promise.all(enabledEngines.map(async engine => {
        const id = (0, _utils2.buildEntityDefinitionId)(engine.type, namespace);
        const {
          definitions: [definition]
        } = await this.entityClient.getEntityDefinitions({
          id,
          includeState: withComponents
        });
        const definitionComponents = this.getComponentFromEntityDefinition(id, definition);
        const entityStoreComponents = await this.getEngineComponentsState(_types.EntityType[engine.type], definition);
        return {
          ...engine,
          components: [...definitionComponents, ...entityStoreComponents]
        };
      }));
      return {
        engines: enginesWithComponents,
        status
      };
    } else {
      return {
        engines: enabledEngines,
        status
      };
    }
  }
  async init(entityType, requestBody, {
    pipelineDebugMode = false
  } = {}) {
    if (!this.options.taskManager) {
      throw new Error('Task Manager is not available');
    }
    const {
      config
    } = this.options;
    await this.riskScoreDataClient.createOrUpdateRiskScoreLatestIndex().catch(e => {
      if (e.meta.body.error.type === 'resource_already_exists_exception') {
        this.options.logger.debug(`Risk score index for ${entityType} already exists, skipping creation.`);
        return;
      }
      throw e;
    });
    const requiresMigration = await this.assetCriticalityMigrationClient.isEcsDataMigrationRequired();
    if (requiresMigration) {
      throw new Error('Asset criticality data migration is required before initializing entity store. If this error persists, please restart Kibana.');
    }
    this.log('info', entityType, `Initializing entity store`);
    this.audit(_actions.EntityEngineActions.INIT, _entity_analytics.EngineComponentResourceEnum.entity_engine, entityType, 'Initializing entity engine');
    const descriptor = await this.engineClient.init(entityType, requestBody);
    this.log('debug', entityType, `Initialized engine saved object`);
    this.asyncSetup(entityType, this.options.taskManager, config, requestBody, pipelineDebugMode).catch(e => this.log('error', entityType, `Error during async setup of entity store: ${e.message}`));
    return descriptor;
  }
  async asyncSetup(entityType, taskManager, config, requestParams, pipelineDebugMode) {
    const setupStartTime = (0, _moment.default)().utc().toISOString();
    const {
      logger,
      namespace,
      appClient,
      dataViewsService
    } = this.options;
    try {
      var _this$options$telemet;
      const defaultIndexPatterns = await (0, _utils2.buildIndexPatternsByEngine)(namespace, entityType, appClient, dataViewsService);
      const options = (0, _merge.merge)(_constants3.defaultOptions, requestParams);
      const description = (0, _engine_description.createEngineDescription)({
        entityType,
        namespace,
        requestParams,
        defaultIndexPatterns,
        config
      });

      // clean up any existing entity store
      await this.delete(entityType, taskManager, {
        deleteData: false,
        deleteEngine: false
      });
      if (this.apiKeyGenerator) {
        await this.apiKeyGenerator.generate();
      }

      // set up the entity manager definition
      const definition = (0, _entity_manager_conversion.convertToEntityManagerDefinition)(description, {
        namespace,
        filter: options.filter
      });
      await this.entityClient.createEntityDefinition({
        definition,
        installOnly: true
      });
      this.log(`debug`, entityType, `Created entity definition`);

      // the index must be in place with the correct mapping before the enrich policy is created
      // this is because the enrich policy will fail if the index does not exist with the correct fields
      await (0, _elasticsearch_assets.createEntityIndexComponentTemplate)(description, this.esClient);
      this.log(`debug`, entityType, `Created entity index component template`);
      await (0, _elasticsearch_assets.createEntityIndex)({
        entityType,
        esClient: this.esClient,
        namespace,
        logger
      });
      this.log(`debug`, entityType, `Created entity index`);

      // Create reset index required by Snapshot task
      await (0, _elasticsearch_assets.createEntityResetIndex)({
        entityType,
        esClient: this.esClient,
        namespace
      });
      this.log(`debug`, entityType, `Created entity reset index`);

      // we must create and execute the enrich policy before the pipeline is created
      // this is because the pipeline will fail if the enrich index does not exist
      await (0, _elasticsearch_assets.createFieldRetentionEnrichPolicy)({
        description,
        options: {
          namespace
        },
        esClient: this.esClient
      });
      this.log(`debug`, entityType, `Created field retention enrich policy`);
      await (0, _elasticsearch_assets.executeFieldRetentionEnrichPolicy)({
        entityType,
        version: description.version,
        options: {
          namespace
        },
        esClient: this.esClient,
        logger
      });
      this.log(`debug`, entityType, `Executed field retention enrich policy`);
      await (0, _elasticsearch_assets.createPlatformPipeline)({
        description,
        options: {
          namespace
        },
        debugMode: pipelineDebugMode,
        logger,
        esClient: this.esClient
      });
      this.log(`debug`, entityType, `Created @platform pipeline`);

      // CRUD Assets
      await (0, _updates_component_template.createEntityUpdatesIndexComponentTemplate)(description, this.esClient);
      this.log(`debug`, entityType, `Created entity updates index component template`);
      await (0, _updates_entity_data_stream.initEntityUpdatesDataStream)(entityType, this.esClient, namespace);
      this.log(`debug`, entityType, `Initialized entity updates data stream`);

      // finally start the entity definition now that everything is in place
      const updated = await this.start(entityType, {
        force: true
      });

      // the task will execute the enrich policy on a schedule
      await (0, _tasks.startEntityStoreFieldRetentionEnrichTask)({
        namespace,
        logger,
        taskManager,
        interval: options.enrichPolicyExecutionInterval
      });
      this.log(`debug`, entityType, `Started entity store field retention enrich task`);

      // this task will continuously refresh the Entity Store indices based on the Data View
      await (0, _tasks.startEntityStoreDataViewRefreshTask)({
        namespace,
        logger,
        taskManager
      });
      this.log(`debug`, entityType, `Started entity store data view refresh task`);

      // this task will create daily snapshots for the historical view
      await (0, _tasks.startEntityStoreSnapshotTask)({
        namespace,
        logger,
        entityType,
        taskManager
      });
      this.log(`debug`, entityType, `Started entity store snapshot task`);
      this.log(`info`, entityType, `Entity store initialized`);
      const setupEndTime = (0, _moment.default)().utc().toISOString();
      const duration = (0, _moment.default)(setupEndTime).diff((0, _moment.default)(setupStartTime), 'seconds');
      (_this$options$telemet = this.options.telemetry) === null || _this$options$telemet === void 0 ? void 0 : _this$options$telemet.reportEvent(_events.ENTITY_ENGINE_INITIALIZATION_EVENT.eventType, {
        duration
      });
      return updated;
    } catch (err) {
      var _this$options$telemet2;
      this.log(`error`, entityType, `Error initializing entity store: ${err.message}`);
      this.audit(_actions.EntityEngineActions.INIT, _entity_analytics.EngineComponentResourceEnum.entity_engine, entityType, 'Failed to initialize entity engine resources', err);
      (_this$options$telemet2 = this.options.telemetry) === null || _this$options$telemet2 === void 0 ? void 0 : _this$options$telemet2.reportEvent(_events.ENTITY_ENGINE_RESOURCE_INIT_FAILURE_EVENT.eventType, {
        error: err.message
      });
      await this.engineClient.update(entityType, {
        status: _constants3.ENGINE_STATUS.ERROR,
        error: {
          message: err.message,
          action: 'init'
        }
      });
      await this.delete(entityType, taskManager, {
        deleteData: true,
        deleteEngine: false
      });
    }
  }
  getComponentFromEntityDefinition(id, definition) {
    if (!definition) {
      return [{
        id,
        installed: false,
        resource: _entity_analytics.EngineComponentResourceEnum.entity_definition
      }];
    }
    if ('state' in definition) {
      const transformHealthToComponentHealth = health => health ? health.toLowerCase() : 'unknown';
      return [{
        id: definition.id,
        installed: definition.state.installed,
        resource: _entity_analytics.EngineComponentResourceEnum.entity_definition
      }, ...definition.state.components.transforms.map(({
        installed,
        stats
      }) => {
        var _stats$health, _stats$health2, _stats$health2$issues;
        return {
          id: (stats === null || stats === void 0 ? void 0 : stats.id) || id,
          resource: _entity_analytics.EngineComponentResourceEnum.transform,
          installed,
          health: transformHealthToComponentHealth(stats === null || stats === void 0 ? void 0 : (_stats$health = stats.health) === null || _stats$health === void 0 ? void 0 : _stats$health.status),
          metadata: stats === null || stats === void 0 ? void 0 : stats.stats,
          errors: stats === null || stats === void 0 ? void 0 : (_stats$health2 = stats.health) === null || _stats$health2 === void 0 ? void 0 : (_stats$health2$issues = _stats$health2.issues) === null || _stats$health2$issues === void 0 ? void 0 : _stats$health2$issues.map(({
            issue,
            details
          }) => ({
            title: issue,
            message: details
          }))
        };
      }), ...definition.state.components.ingestPipelines.map(pipeline => ({
        resource: _entity_analytics.EngineComponentResourceEnum.ingest_pipeline,
        ...pipeline
      })), ...definition.state.components.indexTemplates.map(({
        installed,
        id: templateId
      }) => ({
        id: templateId,
        installed,
        resource: _entity_analytics.EngineComponentResourceEnum.index_template
      }))];
    }
    return [];
  }
  async getExistingEntityDefinition(entityType) {
    const entityDefinitionId = (0, _utils2.buildEntityDefinitionId)(entityType, this.options.namespace);
    const {
      definitions: [definition]
    } = await this.entityClient.getEntityDefinitions({
      id: entityDefinitionId
    });
    if (!definition) {
      throw new Error(`Unable to find entity definition for ${entityType}`);
    }
    return definition;
  }
  async start(entityType, options) {
    const {
      namespace
    } = this.options;
    const descriptor = await this.engineClient.get(entityType);
    if (!(options !== null && options !== void 0 && options.force) && descriptor.status !== _constants3.ENGINE_STATUS.STOPPED) {
      throw new Error(`In namespace ${namespace}: Cannot start Entity engine for ${entityType} when current status is: ${descriptor.status}`);
    }
    this.log('info', entityType, `Starting entity store`);

    // startEntityDefinition requires more fields than the engine descriptor
    // provides so we need to fetch the full entity definition
    const fullEntityDefinition = await this.getExistingEntityDefinition(entityType);
    this.audit(_actions.EntityEngineActions.START, _entity_analytics.EngineComponentResourceEnum.entity_definition, entityType, 'Starting entity definition');
    await this.entityClient.startEntityDefinition(fullEntityDefinition);
    this.log('debug', entityType, `Started entity definition`);
    return this.engineClient.updateStatus(entityType, _constants3.ENGINE_STATUS.STARTED);
  }
  async stop(entityType) {
    const {
      namespace
    } = this.options;
    const descriptor = await this.engineClient.get(entityType);
    if (descriptor.status !== _constants3.ENGINE_STATUS.STARTED) {
      throw new Error(`In namespace ${namespace}: Cannot stop Entity engine for ${entityType} when current status is: ${descriptor.status}`);
    }
    this.log('info', entityType, `Stopping entity store`);

    // stopEntityDefinition requires more fields than the engine descriptor
    // provides so we need to fetch the full entity definition
    const fullEntityDefinition = await this.getExistingEntityDefinition(entityType);
    this.audit(_actions.EntityEngineActions.STOP, _entity_analytics.EngineComponentResourceEnum.entity_definition, entityType, 'Stopping entity definition');
    await this.entityClient.stopEntityDefinition(fullEntityDefinition);
    this.log('debug', entityType, `Stopped entity definition`);
    return this.engineClient.updateStatus(entityType, _constants3.ENGINE_STATUS.STOPPED);
  }
  async get(entityType) {
    return this.engineClient.get(entityType);
  }
  async list() {
    return this.engineClient.list();
  }
  async delete(entityType, taskManager, options = {
    deleteData: false,
    deleteEngine: true
  }) {
    const {
      namespace,
      logger,
      appClient,
      dataViewsService,
      config
    } = this.options;
    const {
      deleteData,
      deleteEngine
    } = options;
    const descriptor = await this.engineClient.maybeGet(entityType);
    const defaultIndexPatterns = await (0, _utils2.buildIndexPatternsByEngine)(namespace, entityType, appClient, dataViewsService);
    const description = (0, _engine_description.createEngineDescription)({
      entityType,
      namespace,
      defaultIndexPatterns,
      config
    });
    if (!description.id) {
      throw new Error(`Unable to find entity definition for ${entityType}`);
    }
    this.log('info', entityType, `Deleting entity store`);
    this.audit(_actions.EntityEngineActions.DELETE, _entity_analytics.EngineComponentResourceEnum.entity_engine, entityType, 'Deleting entity engine');
    try {
      await this.entityClient.deleteEntityDefinition({
        id: description.id,
        deleteData
      })
      // Swallowing the error as it is expected to fail if no entity definition exists
      .catch(e => this.log(`warn`, entityType, `Error deleting entity definition: ${e.message}`));
      this.log('debug', entityType, `Deleted entity definition`);
      await (0, _elasticsearch_assets.deleteEntityIndexComponentTemplate)({
        id: description.id,
        esClient: this.esClient
      });
      this.log('debug', entityType, `Deleted entity index component template`);
      await (0, _elasticsearch_assets.deletePlatformPipeline)({
        description,
        logger,
        esClient: this.esClient
      });
      this.log('debug', entityType, `Deleted platform pipeline`);
      await (0, _elasticsearch_assets.deleteFieldRetentionEnrichPolicy)({
        description,
        options: {
          namespace
        },
        esClient: this.esClient,
        logger
      });
      this.log('debug', entityType, `Deleted field retention enrich policy`);
      await (0, _tasks.removeEntityStoreSnapshotTask)({
        namespace,
        logger,
        entityType,
        taskManager
      });
      this.log('debug', entityType, `Deleted entity store snapshot task`);

      // CRUD Assets
      await (0, _updates_entity_data_stream.deleteEntityUpdatesDataStreams)(entityType, this.esClient, namespace);
      this.log('debug', entityType, `Delete entity updates index`);
      await (0, _updates_component_template.deleteEntityUpdatesIndexComponentTemplate)(description, this.esClient);
      this.log('debug', entityType, `Delete entity updates index`);
      if (deleteData) {
        await (0, _elasticsearch_assets.deleteEntityIndex)({
          entityType,
          esClient: this.esClient,
          namespace,
          logger
        });
        this.log('debug', entityType, `Deleted entity index`);
        await (0, _elasticsearch_assets.deleteAllEntitySnapshotIndices)({
          entityType,
          esClient: this.esClient,
          namespace
        });
        this.log('debug', entityType, `Deleted snapshot indices`);
      }
      await (0, _elasticsearch_assets.deleteEntityResetIndex)({
        entityType,
        esClient: this.esClient,
        namespace
      });
      this.log('debug', entityType, `Deleted reset index`);
      if (descriptor && deleteEngine) {
        await this.engineClient.delete(entityType);
      }
      // if the last engine then stop the task
      const {
        engines
      } = await this.engineClient.list();
      if (engines.length === 0) {
        await (0, _tasks.removeEntityStoreFieldRetentionEnrichTask)({
          namespace,
          logger,
          taskManager
        });
        await (0, _tasks.removeEntityStoreDataViewRefreshTask)({
          namespace,
          logger,
          taskManager
        });
        this.log('debug', entityType, `Deleted entity store field retention and data view refresh tasks`);
      }
      logger.info(`[Entity Store] In namespace ${namespace}: Deleted store for ${entityType}`);
      return {
        deleted: true
      };
    } catch (err) {
      this.log(`error`, entityType, `Error deleting entity store: ${err.message}`);
      this.audit(_actions.EntityEngineActions.DELETE, _entity_analytics.EngineComponentResourceEnum.entity_engine, entityType, 'Failed to delete entity engine', err);
      throw err;
    }
  }
  async searchEntities(params) {
    var _hits$total$value, _hits$total;
    const {
      page,
      perPage,
      sortField,
      sortOrder,
      filterQuery,
      entityTypes
    } = params;
    const index = entityTypes.map(type => (0, _utils2.getEntitiesIndexName)(type, this.options.namespace));
    const from = (page - 1) * perPage;
    const sort = sortField ? [{
      [sortField]: sortOrder
    }] : undefined;
    const query = filterQuery ? JSON.parse(filterQuery) : undefined;
    const response = await this.esClient.search({
      index,
      query,
      size: Math.min(perPage, _constants3.MAX_SEARCH_RESPONSE_SIZE),
      from,
      sort,
      ignore_unavailable: true
    });
    const {
      hits
    } = response;
    const total = typeof hits.total === 'number' ? hits.total : (_hits$total$value = (_hits$total = hits.total) === null || _hits$total === void 0 ? void 0 : _hits$total.value) !== null && _hits$total$value !== void 0 ? _hits$total$value : 0;
    const records = hits.hits.map(hit => {
      const {
        asset,
        ...source
      } = hit._source;
      const assetOverwrite = asset && asset.criticality !== _constants4.CRITICALITY_VALUES.DELETED ? {
        asset: {
          criticality: asset.criticality
        }
      } : {};
      return {
        ...source,
        ...assetOverwrite
      };
    });
    const inspect = {
      dsl: [JSON.stringify({
        index,
        body: query
      }, null, 2)],
      response: [JSON.stringify(response, null, 2)]
    };
    return {
      records,
      total,
      inspect
    };
  }
  async applyDataViewIndices() {
    const {
      logger
    } = this.options;
    logger.debug(`In namespace ${this.options.namespace}: Applying data view indices to the entity store`);
    const {
      engines
    } = await this.engineClient.list();
    if (engines.length === 0) {
      logger.debug(`In namespace ${this.options.namespace}: No entity engines found, skipping data view index application`);
      return {
        successes: [],
        errors: []
      };
    }
    const updateDefinitionPromises = engines.map(async engine => {
      const originalStatus = engine.status;
      const id = (0, _utils2.buildEntityDefinitionId)(engine.type, this.options.namespace);
      const definition = await this.getExistingEntityDefinition(_types.EntityType[engine.type]);
      if (originalStatus === _constants3.ENGINE_STATUS.INSTALLING || originalStatus === _constants3.ENGINE_STATUS.UPDATING) {
        throw new Error(`Error updating entity store: There are changes already in progress for engine ${id}`);
      }
      const defaultIndexPatterns = await (0, _utils2.buildIndexPatternsByEngine)(this.options.namespace, engine.type, this.options.appClient, this.options.dataViewsService);
      const indexPatterns = (0, _utils2.mergeEntityStoreIndices)(defaultIndexPatterns, engine.indexPattern);

      // Skip update if index patterns are the same
      if ((0, _fp.isEqual)(definition.indexPatterns, indexPatterns)) {
        logger.debug(`In namespace ${this.options.namespace}: No data view index changes detected.`);
        return {
          type: engine.type,
          changes: {}
        };
      } else {
        logger.info(`In namespace ${this.options.namespace}: Data view index changes detected, applying changes to entity definition.`);
      }
      const privileges = await this.getEntityStoreSourceIndicesPrivileges(indexPatterns);
      if (!privileges.has_all_required) {
        const missingPrivilegesMsg = (0, _privileges.getMissingPrivilegesErrorMessage)((0, _privileges.getAllMissingPrivileges)(privileges));
        throw new Error(`The current user does not have the required indices privileges for updating the '${engine.type}' entity store.\n${missingPrivilegesMsg}`);
      }

      // Update savedObject status
      await this.engineClient.updateStatus(engine.type, _constants3.ENGINE_STATUS.UPDATING);
      try {
        // Update entity manager definition
        await this.entityClient.updateEntityDefinition({
          id,
          definitionUpdate: {
            ...definition,
            indexPatterns
          }
        });

        // Restore the savedObject status and set the new index pattern
        await this.engineClient.updateStatus(engine.type, originalStatus);
        return {
          type: engine.type,
          changes: {
            indexPatterns
          }
        };
      } catch (error) {
        // Rollback the engine initial status when the update fails
        await this.engineClient.updateStatus(engine.type, originalStatus);
        throw error;
      }
    });
    const updatedDefinitions = await Promise.allSettled(updateDefinitionPromises);
    const updateErrors = updatedDefinitions.filter(_utils2.isPromiseRejected).map(result => result.reason);
    const updateSuccesses = updatedDefinitions.filter(_utils2.isPromiseFulfilled).map(result => result.value);
    return {
      successes: updateSuccesses,
      errors: updateErrors
    };
  }
  getEntityStoreSourceRequiredIndicesPrivileges(securitySolutionIndices) {
    return securitySolutionIndices.reduce((acc, index) => {
      acc[index] = _constants2.ENTITY_STORE_SOURCE_REQUIRED_ES_INDEX_PRIVILEGES;
      return acc;
    }, {});
  }
  log(level, entityType, msg) {
    this.options.logger[level](`[Entity Engine] [entity.${entityType}] [namespace: ${this.options.namespace}] ${msg}`);
  }
  audit(action, resource, entityType, msg, error) {
    var _this$options$auditLo;
    // NOTE: Excluding errors, all auditing events are currently WRITE events, meaning the outcome is always UNKNOWN.
    // This may change in the future, depending on the audit action.
    const outcome = error ? _audit.AUDIT_OUTCOME.FAILURE : _audit.AUDIT_OUTCOME.UNKNOWN;
    const type = action === _actions.EntityEngineActions.CREATE ? _audit.AUDIT_TYPE.CREATION : _actions.EntityEngineActions.DELETE ? _audit.AUDIT_TYPE.DELETION : _audit.AUDIT_TYPE.CHANGE;
    const category = _audit.AUDIT_CATEGORY.DATABASE;
    const message = error ? `${msg}: ${error.message}` : msg;
    const event = {
      message: `[Entity Engine] [entity.${entityType}] ${message}`,
      event: {
        action: `${action}_${entityType}_${resource}`,
        category,
        outcome,
        type
      }
    };
    return (_this$options$auditLo = this.options.auditLogger) === null || _this$options$auditLo === void 0 ? void 0 : _this$options$auditLo.log(event);
  }
  async isCapabilityEnabled(type, capability) {
    const {
      definitions
    } = await this.entityClient.getEntityDefinitions({
      type
    });
    if (definitions.length === 0) {
      return false;
    }
    const capabilities = definitions[0].capabilities || [];
    return capabilities.indexOf(capability) > -1;
  }
}
exports.EntityStoreDataClient = EntityStoreDataClient;