"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.entityDetailsHighlightsRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _local_prompt_object = require("@kbn/elastic-assistant-plugin/server/lib/prompt/local_prompt_object");
var _get_prompt = require("@kbn/elastic-assistant-plugin/server/lib/prompt/get_prompt");
var _highlights = require("../../../../../common/api/entity_analytics/entity_details/highlights.gen");
var _constants = require("../../../../../common/entity_analytics/entity_analytics/constants");
var _types = require("../../../../../common/entity_analytics/types");
var _constants2 = require("../../../../../common/constants");
var _entity_details_highlights_service = require("../entity_details_highlights_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const entityDetailsHighlightsRoute = (router, logger, getStartServices, ml) => {
  router.versioned.post({
    access: 'internal',
    path: _constants.ENTITY_DETAILS_HIGHLIGH_INTERNAL_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants2.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_highlights.EntityDetailsHighlightsRequestBody)
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const entityType = request.body.entityType;
      const entityIdentifier = request.body.entityIdentifier;
      const anonymizationFields = request.body.anonymizationFields;
      const connectorId = request.body.connectorId;
      const entityField = _types.EntityTypeToIdentifierField[entityType];
      const fromDate = request.body.from;
      const toDate = request.body.to;
      const [coreStart] = await getStartServices();
      const securitySolution = await context.securitySolution;
      const actions = await context.actions;
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const spaceId = securitySolution.getSpaceId();
      const coreContext = await context.core;
      const soClient = coreContext.savedObjects.client;
      const riskEngineClient = securitySolution.getRiskEngineDataClient();
      const assetCriticalityClient = securitySolution.getAssetCriticalityDataClient();
      const {
        getRiskScoreData,
        getAssetCriticalityData,
        getVulnerabilityData,
        getAnomaliesData,
        getLocalReplacements
      } = (0, _entity_details_highlights_service.entityDetailsHighlightsServiceFactory)({
        riskEngineClient,
        spaceId,
        logger,
        esClient,
        assetCriticalityClient,
        soClient,
        uiSettingsClient: coreContext.uiSettings.client,
        ml,
        anonymizationFields
      });
      const anonymizedRiskScore = await getRiskScoreData(entityType, entityIdentifier);
      const assetCriticalityAnonymized = await getAssetCriticalityData(entityField, entityIdentifier);
      const {
        vulnerabilitiesAnonymized,
        vulnerabilitiesTotal
      } = await getVulnerabilityData(entityField, entityIdentifier);
      const anomaliesAnonymized = await getAnomaliesData(request, entityField, entityIdentifier, fromDate, toDate);
      const prompt = await (0, _get_prompt.getPrompt)({
        actionsClient: actions.getActionsClient(),
        connectorId,
        promptId: _local_prompt_object.promptDictionary.entityDetailsHighlights,
        promptGroupId: _local_prompt_object.promptGroupId.aiForEntityAnalytics,
        savedObjectsClient: soClient
      });
      return response.ok({
        body: {
          summary: {
            assetCriticality: assetCriticalityAnonymized,
            riskScore: anonymizedRiskScore,
            vulnerabilities: vulnerabilitiesAnonymized !== null && vulnerabilitiesAnonymized !== void 0 ? vulnerabilitiesAnonymized : [],
            vulnerabilitiesTotal,
            // Prevents the UI from displaying the wrong number of vulnerabilities
            anomalies: anomaliesAnonymized
          },
          replacements: getLocalReplacements(entityField, entityIdentifier),
          prompt
        }
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
};
exports.entityDetailsHighlightsRoute = entityDetailsHighlightsRoute;