"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.actionStatusRequestHandler = void 0;
exports.registerActionStatusRoutes = registerActionStatusRoutes;
var _error_handler = require("../error_handler");
var _stringify = require("../../utils/stringify");
var _endpoint = require("../../../../common/api/endpoint");
var _constants = require("../../../../common/endpoint/constants");
var _services = require("../../services");
var _with_endpoint_authz = require("../with_endpoint_authz");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Registers routes for checking status of actions
 */
function registerActionStatusRoutes(router, endpointContext) {
  // Summary of action status for a given list of endpoints
  router.versioned.get({
    access: 'public',
    path: _constants.ACTION_STATUS_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.ActionStatusRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canReadSecuritySolution']
  }, endpointContext.logFactory.get('hostIsolationStatus'), actionStatusRequestHandler(endpointContext)));
}
const actionStatusRequestHandler = function (endpointContext) {
  const logger = endpointContext.logFactory.get('actionStatusApi');
  return async (context, req, res) => {
    logger.debug(() => `Retrieving action status for: ${(0, _stringify.stringify)(req.query.agent_ids)}`);
    try {
      const spaceId = (await context.securitySolution).getSpaceId();
      const agentIDs = Array.isArray(req.query.agent_ids) ? [...new Set(req.query.agent_ids)] : [req.query.agent_ids];
      await endpointContext.service.getInternalFleetServices(spaceId).ensureInCurrentSpace({
        agentIds: agentIDs
      });
      const response = await (0, _services.getPendingActionsSummary)(endpointContext.service, spaceId, agentIDs);
      return res.ok({
        body: {
          data: response
        }
      });
    } catch (error) {
      return (0, _error_handler.errorHandler)(logger, res, error);
    }
  };
};
exports.actionStatusRequestHandler = actionStatusRequestHandler;