"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.exploreIndices = void 0;
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Converts a createLlmInstance result to a ScopedModel for use with onechat utilities
 */
const convertToScopedModel = async (createLlmInstance, inference, request, connectorId) => {
  // Get the chat model from createLlmInstance
  const chatModel = await createLlmInstance();

  // Get the inference client
  const inferenceClient = inference.getClient({
    request,
    bindTo: {
      connectorId
    }
  });

  // Get the connector metadata
  const connector = await inferenceClient.getConnectorById(connectorId);

  // Assemble into ScopedModel
  return {
    chatModel: chatModel,
    // Type assertion needed for compatibility
    connector,
    inferenceClient
  };
};
const exploreIndices = ({
  esClient,
  createLlmInstance,
  inference,
  request,
  connectorId
}) => {
  return async state => {
    const {
      input
    } = state;
    if (!input) {
      throw new Error('Input is required for index exploration');
    }
    const {
      query,
      limit = 10,
      indexPattern = '*'
    } = input;

    // Convert createLlmInstance result to ScopedModel for onechat compatibility
    const model = await convertToScopedModel(createLlmInstance, inference, request, connectorId);

    // Use the indexExplorer utility from onechat
    const response = await (0, _onechatGenaiUtils.indexExplorer)({
      nlQuery: query,
      indexPattern,
      limit,
      esClient,
      model
    });

    // Transform the response to match our state structure
    const discoveredResources = response.resources.map(resource => ({
      type: resource.type,
      name: resource.name,
      reason: resource.reason
    }));

    // Extract index pattern names for the shortlist function
    const indexPatterns = discoveredResources.map(resource => resource.name);
    return {
      discoveredResources,
      indexPatterns,
      input: {
        question: input.query,
        indexPattern: input.indexPattern
      }
    };
  };
};
exports.exploreIndices = exploreIndices;