"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.XmlParser = void 0;
var _xml2js = _interopRequireDefault(require("xml2js"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class XmlParser {
  constructor(xml) {
    this.xml = xml;
  }
  async parse() {
    return _xml2js.default.parseStringPromise(this.xml, {
      explicitArray: true
    });
  }
  findAllDeep(source, elementName) {
    const results = [];
    if (typeof source !== 'object' || source === null) {
      return results;
    }
    if (elementName in source) {
      const element = source[elementName];
      if (Array.isArray(element)) {
        results.push(...element);
      } else if (element) {
        results.push(element);
      }
    }

    // Search recursively in all properties (but skip children of found elements)
    for (const key of Object.keys(source)) {
      if (key === elementName) {
        // Skip the element we already processed above
      } else {
        const value = source[key];
        if (Array.isArray(value)) {
          for (const item of value) {
            const childResults = this.findAllDeep(item, elementName);
            results.push(...childResults);
          }
        } else if (typeof value === 'object' && value !== null) {
          const childResults = this.findAllDeep(value, elementName);
          results.push(...childResults);
        }
      }
    }
    return results;
  }

  /** Unified deep search method (equivalent to XML's .// XPath expressions) */
  findDeep(source, elementName, attrName, attrValue) {
    if (typeof source !== 'object' || source === null) {
      return undefined;
    }
    // Check if the element exists at this level
    if (elementName in source) {
      const element = source[elementName];

      // If no attribute filtering is needed, return the element
      if (!attrName || !attrValue) {
        return element;
      }

      // If attribute filtering is needed, check if it's an array of elements
      if (Array.isArray(element)) {
        for (const item of element) {
          if (item.$ && item.$[attrName] === attrValue) {
            return item;
          }
        }
      }
    }
    for (const key of Object.keys(source)) {
      const value = source[key];
      if (Array.isArray(value)) {
        for (const item of value) {
          const result = this.findDeep(item, elementName, attrName, attrValue);
          if (result !== undefined) {
            return result;
          }
        }
      } else if (typeof value === 'object' && value !== null) {
        const result = this.findDeep(value, elementName, attrName, attrValue);
        if (result !== undefined) {
          return result;
        }
      }
    }
    return undefined;
  }
  findDeepValue(source, elementName, attrName) {
    if (typeof source !== 'object' || source === null) {
      return undefined;
    }
    const el = this.findDeep(source, elementName);

    // If attribute filtering is needed, check if it's an array of elements
    if (Array.isArray(el)) {
      for (const item of el) {
        return item.$ && attrName in item.$ ? item.$[attrName] : undefined;
      }
    } else if (typeof el === 'object' && el !== null) {
      return el.$ && attrName in el.$ ? el.$[attrName] : undefined;
    }
    return undefined;
  }
}
exports.XmlParser = XmlParser;